/*-
 * SPDX-License-Identifier: BSD-2-Clause
 *
 * Copyright (c) 2001 Atsushi Onoe
 * Copyright (c) 2002-2009 Sam Leffler, Errno Consulting
 * All rights reserved.
 * Copyright (c) 2020-2025 The FreeBSD Foundation
 *
 * Portions of this software were developed by Björn Zeeb
 * under sponsorship from the FreeBSD Foundation.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#ifndef _NET80211_IEEE80211_H_
#define _NET80211_IEEE80211_H_

#include <sys/types.h>

#ifndef _KERNEL
#include <stdbool.h>
#endif

/*
 * 802.11 protocol definitions.
 */

#define	IEEE80211_ADDR_LEN	6		/* size of 802.11 address */
/* is 802.11 address multicast/broadcast? */
#define	IEEE80211_IS_MULTICAST(_a)	(*(_a) & 0x01)

#ifdef _KERNEL
extern const uint8_t ieee80211broadcastaddr[];
#endif

typedef uint16_t ieee80211_seq;

/* IEEE 802.11 PLCP header */
struct ieee80211_plcp_hdr {
	uint16_t	i_sfd;
	uint8_t		i_signal;
	uint8_t		i_service;
	uint16_t	i_length;
	uint16_t	i_crc;
} __packed;

#define IEEE80211_PLCP_SFD      0xF3A0 
#define IEEE80211_PLCP_SERVICE  0x00
#define IEEE80211_PLCP_SERVICE_LOCKED	0x04
#define IEEE80211_PLCL_SERVICE_PBCC	0x08
#define IEEE80211_PLCP_SERVICE_LENEXT5	0x20
#define IEEE80211_PLCP_SERVICE_LENEXT6	0x40
#define IEEE80211_PLCP_SERVICE_LENEXT7	0x80

/*
 * generic definitions for IEEE 802.11 frames
 */
struct ieee80211_frame {
	uint8_t		i_fc[2];
	uint8_t		i_dur[2];
	uint8_t		i_addr1[IEEE80211_ADDR_LEN];
	uint8_t		i_addr2[IEEE80211_ADDR_LEN];
	uint8_t		i_addr3[IEEE80211_ADDR_LEN];
	uint8_t		i_seq[2];
	/* possibly followed by addr4[IEEE80211_ADDR_LEN]; */
	/* see below */
} __packed;

struct ieee80211_qosframe {
	uint8_t		i_fc[2];
	uint8_t		i_dur[2];
	uint8_t		i_addr1[IEEE80211_ADDR_LEN];
	uint8_t		i_addr2[IEEE80211_ADDR_LEN];
	uint8_t		i_addr3[IEEE80211_ADDR_LEN];
	uint8_t		i_seq[2];
	uint8_t		i_qos[2];
	/* possibly followed by addr4[IEEE80211_ADDR_LEN]; */
	/* see below */
} __packed;

struct ieee80211_qoscntl {
	uint8_t		i_qos[2];
};

struct ieee80211_frame_addr4 {
	uint8_t		i_fc[2];
	uint8_t		i_dur[2];
	uint8_t		i_addr1[IEEE80211_ADDR_LEN];
	uint8_t		i_addr2[IEEE80211_ADDR_LEN];
	uint8_t		i_addr3[IEEE80211_ADDR_LEN];
	uint8_t		i_seq[2];
	uint8_t		i_addr4[IEEE80211_ADDR_LEN];
} __packed;

struct ieee80211_qosframe_addr4 {
	uint8_t		i_fc[2];
	uint8_t		i_dur[2];
	uint8_t		i_addr1[IEEE80211_ADDR_LEN];
	uint8_t		i_addr2[IEEE80211_ADDR_LEN];
	uint8_t		i_addr3[IEEE80211_ADDR_LEN];
	uint8_t		i_seq[2];
	uint8_t		i_addr4[IEEE80211_ADDR_LEN];
	uint8_t		i_qos[2];
} __packed;

#define	IEEE80211_FC0_VERSION_MASK		0x03
#define	IEEE80211_FC0_VERSION_SHIFT		0
#define	IEEE80211_FC0_VERSION_0			0x00
#define	IEEE80211_FC0_TYPE_MASK			0x0c
#define	IEEE80211_FC0_TYPE_SHIFT		2
#define	IEEE80211_FC0_TYPE_MGT			0x00	/* Management */
#define	IEEE80211_FC0_TYPE_CTL			0x04	/* Control */
#define	IEEE80211_FC0_TYPE_DATA			0x08	/* Data */
#define	IEEE80211_FC0_TYPE_EXT			0x0c	/* Extension */

#define	IEEE80211_FC0_SUBTYPE_MASK		0xf0
#define	IEEE80211_FC0_SUBTYPE_SHIFT		4
/* 802.11-2020 Table 9-1-Valid type and subtype combinations */
/* For type 00 Management (IEEE80211_FC0_TYPE_MGT) */
#define	IEEE80211_FC0_SUBTYPE_ASSOC_REQ		0x00	/* Association Request */
#define	IEEE80211_FC0_SUBTYPE_ASSOC_RESP	0x10	/* Association Response */
#define	IEEE80211_FC0_SUBTYPE_REASSOC_REQ	0x20	/* Reassociation Request */
#define	IEEE80211_FC0_SUBTYPE_REASSOC_RESP	0x30	/* Reassociation Response */
#define	IEEE80211_FC0_SUBTYPE_PROBE_REQ		0x40	/* Probe Request */
#define	IEEE80211_FC0_SUBTYPE_PROBE_RESP	0x50	/* Probe Response */
#define	IEEE80211_FC0_SUBTYPE_TIMING_ADV	0x60	/* Timing Advertisement */
/* 0111 Reserved				0x70 */
#define	IEEE80211_FC0_SUBTYPE_BEACON		0x80	/* Beacon */
#define	IEEE80211_FC0_SUBTYPE_ATIM		0x90	/* ATIM */
#define	IEEE80211_FC0_SUBTYPE_DISASSOC		0xa0	/* Disassociation */
#define	IEEE80211_FC0_SUBTYPE_AUTH		0xb0	/* Authentication */
#define	IEEE80211_FC0_SUBTYPE_DEAUTH		0xc0	/* Deauthentication */
#define	IEEE80211_FC0_SUBTYPE_ACTION		0xd0	/* Action */
#define	IEEE80211_FC0_SUBTYPE_ACTION_NOACK	0xe0	/* Action No Ack */
/* 1111 Reserved				0xf0 */
/* For type 01 Control (IEEE80211_FC0_TYPE_CTL) */
/* 0000-0001 Reserved				0x00-0x10 */
#define	IEEE80211_FC0_SUBTYPE_TRIGGER		0x20	/* Trigger, 80211ax-2021 */
#define	IEEE80211_FC0_SUBTYPE_TACK		0x30	/* TACK */
#define	IEEE80211_FC0_SUBTYPE_BF_REPORT_POLL	0x40	/* Beamforming Report Poll */
#define	IEEE80211_FC0_SUBTYPE_VHT_HE_NDP	0x50	/* VHT/HE NDP Announcement, 80211ac/ax-2013/2021 */
#define	IEEE80211_FC0_SUBTYPE_CTL_EXT		0x60	/* Control Frame Extension */
#define	IEEE80211_FC0_SUBTYPE_CONTROL_WRAP	0x70	/* Control Wrapper */
#define	IEEE80211_FC0_SUBTYPE_BAR		0x80	/* Block Ack Request (BlockAckReq) */
#define	IEEE80211_FC0_SUBTYPE_BA		0x90	/* Block Ack (BlockAck) */
#define	IEEE80211_FC0_SUBTYPE_PS_POLL		0xa0	/* PS-Poll */
#define	IEEE80211_FC0_SUBTYPE_RTS		0xb0	/* RTS */
#define	IEEE80211_FC0_SUBTYPE_CTS		0xc0	/* CTS */
#define	IEEE80211_FC0_SUBTYPE_ACK		0xd0	/* Ack */
#define	IEEE80211_FC0_SUBTYPE_CF_END		0xe0	/* CF-End */
#define	IEEE80211_FC0_SUBTYPE_CF_END_ACK	0xf0	/* 1111 Reserved - what was CF_END_ACK? */
/* For type 10 Data (IEEE80211_FC0_TYPE_DATA) */
#define	IEEE80211_FC0_SUBTYPE_DATA		0x00	/* Data */
/* 0001-0011 Reserved				0x10-0x30 */	/* Were: CF_ACK, CF_POLL, CF_ACPL */
#define	IEEE80211_FC0_SUBTYPE_NODATA		0x40	/* Null */
/* 0101-0111 Reserved				0x50-0x70 */	/* Were: CFACK, CFPOLL, CF_ACK_CF_ACK */
#define	IEEE80211_FC0_SUBTYPE_QOS_MASK_ANY	0x80	/* QoS mask - matching any subtypes 8..15 */
#define	IEEE80211_FC0_SUBTYPE_QOS_DATA		0x80	/* QoS Data */
#define	IEEE80211_FC0_SUBTYPE_QOS_DATA_CFACK	0x90	/* QoS Data +CF-Ack */
#define	IEEE80211_FC0_SUBTYPE_QOS_DATA_CFPOLL	0xa0	/* QoS Data +CF-Poll */
#define	IEEE80211_FC0_SUBTYPE_QOS_DATA_CFACKPOLL 0xb0	/* QoS Data +CF-Ack +CF-Poll */
#define	IEEE80211_FC0_SUBTYPE_QOS_NULL		0xc0	/* QoS Null */
/* 1101 Reserved				0xd0 */
#define	IEEE80211_FC0_SUBTYPE_QOS_CFPOLL	0xe0	/* QoS CF-Poll */
#define	IEEE80211_FC0_SUBTYPE_QOS_CFACKPOLL	0xf0	/* QoS CF-Ack +CF-Poll */
/* For type 11 Extension (IEEE80211_FC0_TYPE_EXT) */
#define	IEEE80211_FC0_SUBTYPE_DMG_BEACON	0x00	/* DMG Beacon */
#define	IEEE80211_FC0_SUBTYPE_S1G_BEACON	0x10	/* S1G Beacon */
/* 0010-1111 Reserved				0x20-0xff */

/* 802.11-2020 Table 9-2-Control Frame Extension */
/* Reusing B11..B8, part of FC1 */
#define	IEEE80211_CTL_EXT_SECTOR_ACK		0x00	/* Sector Ack, 80211ay-2021 */
#define	IEEE80211_CTL_EXT_BA_SCHED		0x01	/* Block Ack Schedule, 80211ay-2021 */
#define	IEEE80211_CTL_EXT_POLL			0x02	/* Poll */
#define	IEEE80211_CTL_EXT_SPR			0x03	/* SPR */
#define	IEEE80211_CTL_EXT_GRANT			0x04	/* Grant */
#define	IEEE80211_CTL_EXT_DMG_CTS		0x05	/* DMG CTS */
#define	IEEE80211_CTL_EXT_DMG_DTS		0x06	/* DMG DTS */
#define	IEEE80211_CTL_EXT_GRANT_ACK		0x07	/* Grant Ack */
#define	IEEE80211_CTL_EXT_SSW			0x08	/* SSW */
#define	IEEE80211_CTL_EXT_SSW_FBACK		0x09	/* SSW-Feedback */
#define	IEEE80211_CTL_EXT_SSW_ACK		0x0a	/* SSW-Ack */
#define	IEEE80211_CTL_EXT_TDD_BF		0x0b	/* TDD Beamforming, 80211ay-2021 */
/* 1100-1111 Reserved				0xc-0xf */

/* Check the version field */
#define	IEEE80211_IS_FC0_CHECK_VER(wh, v)			\
	(((wh)->i_fc[0] & IEEE80211_FC0_VERSION_MASK) == (v))

/* Check the version and type field */
#define	IEEE80211_IS_FC0_CHECK_VER_TYPE(wh, v, t)			\
	(((((wh)->i_fc[0] & IEEE80211_FC0_VERSION_MASK) == (v))) &&	\
	  (((wh)->i_fc[0] & IEEE80211_FC0_TYPE_MASK) == (t)))

/* Check the version, type and subtype field */
#define	IEEE80211_IS_FC0_CHECK_VER_TYPE_SUBTYPE(wh, v, t, st)		\
	(((((wh)->i_fc[0] & IEEE80211_FC0_VERSION_MASK) == (v))) &&	\
	  (((wh)->i_fc[0] & IEEE80211_FC0_TYPE_MASK) == (t)) &&		\
	  (((wh)->i_fc[0] & IEEE80211_FC0_SUBTYPE_MASK) == (st)))

#define	IEEE80211_IS_MGMT(wh)						\
	(IEEE80211_IS_FC0_CHECK_VER_TYPE(wh, IEEE80211_FC0_VERSION_0,	\
	 IEEE80211_FC0_TYPE_MGT))
#define	IEEE80211_IS_CTL(wh)					\
	(IEEE80211_IS_FC0_CHECK_VER_TYPE(wh, IEEE80211_FC0_VERSION_0,	\
	 IEEE80211_FC0_TYPE_CTL))
#define	IEEE80211_IS_DATA(wh)					\
	(IEEE80211_IS_FC0_CHECK_VER_TYPE(wh, IEEE80211_FC0_VERSION_0,	\
	 IEEE80211_FC0_TYPE_DATA))
#define	IEEE80211_IS_EXT(wh)					\
	(IEEE80211_IS_FC0_CHECK_VER_TYPE(wh, IEEE80211_FC0_VERSION_0,	\
	 IEEE80211_FC0_TYPE_EXT))

/* Management frame types */

#define	IEEE80211_IS_MGMT_BEACON(wh)			\
	(IEEE80211_IS_FC0_CHECK_VER_TYPE_SUBTYPE(wh,	\
	 IEEE80211_FC0_VERSION_0,			\
	 IEEE80211_FC0_TYPE_MGT,			\
	 IEEE80211_FC0_SUBTYPE_BEACON))

#define	IEEE80211_IS_MGMT_PROBE_RESP(wh)		\
	(IEEE80211_IS_FC0_CHECK_VER_TYPE_SUBTYPE(wh,	\
	 IEEE80211_FC0_VERSION_0,			\
	 IEEE80211_FC0_TYPE_MGT,			\
	 IEEE80211_FC0_SUBTYPE_PROBE_RESP))

#define	IEEE80211_IS_MGMT_ACTION(wh)		\
	(IEEE80211_IS_FC0_CHECK_VER_TYPE_SUBTYPE(wh,	\
	 IEEE80211_FC0_VERSION_0,			\
	 IEEE80211_FC0_TYPE_MGT,			\
	 IEEE80211_FC0_SUBTYPE_ACTION))

/* Control frame types */

#define	IEEE80211_IS_CTL_PS_POLL(wh)			\
	(IEEE80211_IS_FC0_CHECK_VER_TYPE_SUBTYPE(wh,	\
	 IEEE80211_FC0_VERSION_0,			\
	 IEEE80211_FC0_TYPE_CTL,			\
	 IEEE80211_FC0_SUBTYPE_PS_POLL))

#define	IEEE80211_IS_CTL_BAR(wh)			\
	(IEEE80211_IS_FC0_CHECK_VER_TYPE_SUBTYPE(wh,	\
	 IEEE80211_FC0_VERSION_0,			\
	 IEEE80211_FC0_TYPE_CTL,			\
	 IEEE80211_FC0_SUBTYPE_BAR))

/* Data frame types */

/*
 * Return true if the frame is any of the QOS frame types, not just
 * data frames.  Matching on the IEEE80211_FC0_SUBTYPE_QOS_ANY bit
 * being set also matches on subtypes 8..15.
 */
#define	IEEE80211_IS_QOS_ANY(wh)					\
	((IEEE80211_IS_FC0_CHECK_VER_TYPE(wh, IEEE80211_FC0_VERSION_0,	\
	 IEEE80211_FC0_TYPE_DATA)) &&					\
	 ((wh)->i_fc[0] & IEEE80211_FC0_SUBTYPE_QOS_MASK_ANY))

/*
 * Return true if this frame is QOS data, and only QOS data.
 */
#define	IEEE80211_IS_QOSDATA(wh)			\
	(IEEE80211_IS_FC0_CHECK_VER_TYPE_SUBTYPE(wh,	\
	 IEEE80211_FC0_VERSION_0,			\
	 IEEE80211_FC0_TYPE_DATA,			\
	 IEEE80211_FC0_SUBTYPE_QOS_DATA))

/*
 * Return true if this frame is a QoS NULL data frame.
 */
#define	IEEE80211_IS_QOS_NULL(wh)			\
	(IEEE80211_IS_FC0_CHECK_VER_TYPE_SUBTYPE(wh,	\
	 IEEE80211_FC0_VERSION_0,			\
	 IEEE80211_FC0_TYPE_DATA,			\
	 IEEE80211_FC0_SUBTYPE_QOS_NULL))


#define	IEEE80211_FC1_DIR_MASK			0x03
#define	IEEE80211_FC1_DIR_NODS			0x00	/* STA->STA */
#define	IEEE80211_FC1_DIR_TODS			0x01	/* STA->AP  */
#define	IEEE80211_FC1_DIR_FROMDS		0x02	/* AP ->STA */
#define	IEEE80211_FC1_DIR_DSTODS		0x03	/* AP ->AP  */

#define	IEEE80211_IS_DSTODS(wh) \
	(((wh)->i_fc[1] & IEEE80211_FC1_DIR_MASK) == IEEE80211_FC1_DIR_DSTODS)

#define	IEEE80211_FC1_MORE_FRAG			0x04
#define	IEEE80211_FC1_RETRY			0x08
#define	IEEE80211_FC1_PWR_MGT			0x10
#define	IEEE80211_FC1_MORE_DATA			0x20
#define	IEEE80211_FC1_PROTECTED			0x40
#define	IEEE80211_FC1_ORDER			0x80

#define	IEEE80211_IS_PROTECTED(wh) \
	((wh)->i_fc[1] & IEEE80211_FC1_PROTECTED)

#define IEEE80211_HAS_SEQ(type, subtype) \
	((type) != IEEE80211_FC0_TYPE_CTL && \
	!((type) == IEEE80211_FC0_TYPE_DATA && \
	 ((subtype) & IEEE80211_FC0_SUBTYPE_QOS_NULL) == \
		      IEEE80211_FC0_SUBTYPE_QOS_NULL))
#define	IEEE80211_SEQ_FRAG_MASK			0x000f
#define	IEEE80211_SEQ_FRAG_SHIFT		0
#define	IEEE80211_SEQ_SEQ_MASK			0xfff0
#define	IEEE80211_SEQ_SEQ_SHIFT			4
#define	IEEE80211_SEQ_RANGE			4096

#define	IEEE80211_SEQ_ADD(seq, incr) \
	(((seq) + (incr)) & (IEEE80211_SEQ_RANGE-1))
#define	IEEE80211_SEQ_INC(seq)	IEEE80211_SEQ_ADD(seq,1)
#define	IEEE80211_SEQ_SUB(a, b) \
	(((a) + IEEE80211_SEQ_RANGE - (b)) & (IEEE80211_SEQ_RANGE-1))

#define	IEEE80211_SEQ_BA_RANGE			2048	/* 2^11 */
#define	IEEE80211_SEQ_BA_BEFORE(a, b) \
	(IEEE80211_SEQ_SUB(b, a+1) < IEEE80211_SEQ_BA_RANGE-1)

#define	IEEE80211_NWID_LEN			32
#define	IEEE80211_MESHID_LEN			32

#define	IEEE80211_QOS_CTL_LEN			2

#define	IEEE80211_QOS_TXOP			0x00ff
/* bit 8 is reserved */
#define	IEEE80211_QOS_AMSDU			0x80
#define	IEEE80211_QOS_AMSDU_S			7
#define	IEEE80211_QOS_ACKPOLICY			0x60
#define	IEEE80211_QOS_ACKPOLICY_S		5
#define	IEEE80211_QOS_ACKPOLICY_NOACK		0x20	/* No ACK required */
#define	IEEE80211_QOS_ACKPOLICY_BA		0x60	/* Block ACK */
#define	IEEE80211_QOS_EOSP			0x10	/* EndOfService Period*/
#define	IEEE80211_QOS_EOSP_S			4
#define	IEEE80211_QOS_TID			0x0f
/* qos[1] byte used for all frames sent by mesh STAs in a mesh BSS */
#define IEEE80211_QOS_MC			0x01	/* Mesh control */
/* Mesh power save level*/
#define IEEE80211_QOS_MESH_PSL			0x02
/* Mesh Receiver Service Period Initiated */
#define IEEE80211_QOS_RSPI			0x04
/* bits 11 to 15 reserved */

/* does frame have QoS sequence control data */
#define	IEEE80211_QOS_HAS_SEQ(wh) \
	(((wh)->i_fc[0] & \
	  (IEEE80211_FC0_TYPE_MASK | IEEE80211_FC0_SUBTYPE_QOS_DATA)) == \
	  (IEEE80211_FC0_TYPE_DATA | IEEE80211_FC0_SUBTYPE_QOS_DATA))

/*
 * WME/802.11e information element.
 */
struct ieee80211_wme_info {
	uint8_t		wme_id;		/* IEEE80211_ELEMID_VENDOR */
	uint8_t		wme_len;	/* length in bytes */
	uint8_t		wme_oui[3];	/* 0x00, 0x50, 0xf2 */
	uint8_t		wme_type;	/* OUI type */
	uint8_t		wme_subtype;	/* OUI subtype */
	uint8_t		wme_version;	/* spec revision */
	uint8_t		wme_info;	/* QoS info */
} __packed;

/*
 * WME/802.11e Tspec Element
 */
struct ieee80211_wme_tspec {
	uint8_t		ts_id;
	uint8_t		ts_len;
	uint8_t		ts_oui[3];
	uint8_t		ts_oui_type;
	uint8_t		ts_oui_subtype;
	uint8_t		ts_version;
	uint8_t		ts_tsinfo[3];
	uint8_t		ts_nom_msdu[2];
	uint8_t		ts_max_msdu[2];
	uint8_t		ts_min_svc[4];
	uint8_t		ts_max_svc[4];
	uint8_t		ts_inactv_intv[4];
	uint8_t		ts_susp_intv[4];
	uint8_t		ts_start_svc[4];
	uint8_t		ts_min_rate[4];
	uint8_t		ts_mean_rate[4];
	uint8_t		ts_max_burst[4];
	uint8_t		ts_min_phy[4];
	uint8_t		ts_peak_rate[4];
	uint8_t		ts_delay[4];
	uint8_t		ts_surplus[2];
	uint8_t		ts_medium_time[2];
} __packed;

/*
 * WME AC parameter field
 */
struct ieee80211_wme_acparams {
	uint8_t		acp_aci_aifsn;
	uint8_t		acp_logcwminmax;
	uint16_t	acp_txop;
} __packed;

#define WME_NUM_AC		4	/* 4 AC categories */
#define	WME_NUM_TID		16	/* 16 tids */

#define WME_PARAM_ACI		0x60	/* Mask for ACI field */
#define WME_PARAM_ACI_S		5	/* Shift for ACI field */
#define WME_PARAM_ACM		0x10	/* Mask for ACM bit */
#define WME_PARAM_ACM_S		4	/* Shift for ACM bit */
#define WME_PARAM_AIFSN		0x0f	/* Mask for aifsn field */
#define WME_PARAM_AIFSN_S	0	/* Shift for aifsn field */
#define WME_PARAM_LOGCWMIN	0x0f	/* Mask for CwMin field (in log) */
#define WME_PARAM_LOGCWMIN_S	0	/* Shift for CwMin field */
#define WME_PARAM_LOGCWMAX	0xf0	/* Mask for CwMax field (in log) */
#define WME_PARAM_LOGCWMAX_S	4	/* Shift for CwMax field */

#define WME_AC_TO_TID(_ac) (       \
	((_ac) == WME_AC_VO) ? 6 : \
	((_ac) == WME_AC_VI) ? 5 : \
	((_ac) == WME_AC_BK) ? 1 : \
	0)

#define TID_TO_WME_AC(_tid) (      \
	((_tid) == 0 || (_tid) == 3) ? WME_AC_BE : \
	((_tid) < 3) ? WME_AC_BK : \
	((_tid) < 6) ? WME_AC_VI : \
	WME_AC_VO)

/*
 * WME Parameter Element
 */
struct ieee80211_wme_param {
	uint8_t		param_id;
	uint8_t		param_len;
	uint8_t		param_oui[3];
	uint8_t		param_oui_type;
	uint8_t		param_oui_subtype;
	uint8_t		param_version;
	uint8_t		param_qosInfo;
#define	WME_QOSINFO_COUNT	0x0f	/* Mask for param count field */
	uint8_t		param_reserved;
	struct ieee80211_wme_acparams	params_acParams[WME_NUM_AC];
} __packed;

/*
 * WME U-APSD qos info field defines
 */
#define	WME_CAPINFO_UAPSD_EN                    0x00000080
#define	WME_CAPINFO_UAPSD_VO                    0x00000001
#define	WME_CAPINFO_UAPSD_VI                    0x00000002
#define	WME_CAPINFO_UAPSD_BK                    0x00000004
#define	WME_CAPINFO_UAPSD_BE                    0x00000008
#define	WME_CAPINFO_UAPSD_ACFLAGS_SHIFT         0
#define	WME_CAPINFO_UAPSD_ACFLAGS_MASK          0xF
#define	WME_CAPINFO_UAPSD_MAXSP_SHIFT           5
#define	WME_CAPINFO_UAPSD_MAXSP_MASK            0x3
#define	WME_CAPINFO_IE_OFFSET                   8
#define	WME_UAPSD_MAXSP(_qosinfo)				\
	    (((_qosinfo) >> WME_CAPINFO_UAPSD_MAXSP_SHIFT) &	\
	    WME_CAPINFO_UAPSD_MAXSP_MASK)
#define	WME_UAPSD_AC_ENABLED(_ac, _qosinfo)			\
	    ((1 << (3 - (_ac))) & (				\
	    ((_qosinfo) >> WME_CAPINFO_UAPSD_ACFLAGS_SHIFT) &	\
	    WME_CAPINFO_UAPSD_ACFLAGS_MASK))

/*
 * Management Notification Frame
 */
struct ieee80211_mnf {
	uint8_t		mnf_category;
	uint8_t		mnf_action;
	uint8_t		mnf_dialog;
	uint8_t		mnf_status;
} __packed;
#define	MNF_SETUP_REQ	0
#define	MNF_SETUP_RESP	1
#define	MNF_TEARDOWN	2

/* 
 * 802.11n Management Action Frames 
 */
/* generic frame format */
struct ieee80211_action {
	uint8_t		ia_category;
	uint8_t		ia_action;
} __packed;

/* 80211-2020 Table 9-51-Category values */
#define	IEEE80211_ACTION_CAT_SM			0	/* 9.6.2 Spectrum Management */
#define	IEEE80211_ACTION_CAT_QOS		1	/* 9.6.3 QoS */
/* Reserved					2	was IEEE80211_ACTION_CAT_DLS */
#define	IEEE80211_ACTION_CAT_BA			3	/* 9.6.4 Block Ack */
#define	IEEE80211_ACTION_CAT_PUBLIC		4	/* 9.6.7 Public */
#define	IEEE80211_ACTION_CAT_RADIO_MEASUREMENT	5	/* 9.6.6 Radio Measurement */
#define	IEEE80211_ACTION_CAT_FAST_BSS_TRANSITION 6	/* 9.6.8 Fast BSS Transition */
#define	IEEE80211_ACTION_CAT_HT			7	/* 9.6.11 HT */
#define	IEEE80211_ACTION_CAT_SA_QUERY		8	/* 9.6.9 SA Query */
#define	IEEE80211_ACTION_CAT_PROTECTED_DUAL_OF_PUBLIC_ACTION 9 /* 9.6.10 Protected Dual of Public Action */
#define	IEEE80211_ACTION_CAT_WNM		10	/* 9.6.13 WNM */
#define	IEEE80211_ACTION_CAT_UNPROTECTED_WNM	11	/* 9.6.14 Unprotected WNM */
#define	IEEE80211_ACTION_CAT_TDLS		12	/* 9.6.12 TDLS */
#define	IEEE80211_ACTION_CAT_MESH		13	/* 9.6.16 Mesh */
#define	IEEE80211_ACTION_CAT_MULTIHOP		14	/* 9.6.17 Multihop */
#define	IEEE80211_ACTION_CAT_SELF_PROT		15	/* 9.6.15 Self-protected */
#define	IEEE80211_ACTION_CAT_DMG		16	/* 9.6.19 DMG */
/* Reserved					17	(R)Wi-Fi Alliance */
#define	IEEE80211_ACTION_CAT_FAST_SESSION_TRANSFER 18	/* 9.6.20 Fast Session Transfer */
#define	IEEE80211_ACTION_CAT_ROBUST_AV_STREAMING 19	/* 9.6.18 Robust AV Streaming */
#define	IEEE80211_ACTION_CAT_UNPROTECTED_DMG	20	/* 9.6.21 Unprotected DMG */
#define	IEEE80211_ACTION_CAT_VHT		21	/* 9.6.22 VHT */
#define	IEEE80211_ACTION_CAT_UNPROTECTED_S1G	22	/* 9.6.24 Unprotected S1G */
#define	IEEE80211_ACTION_CAT_S1G		23	/* 9.6.25 S1G */
#define	IEEE80211_ACTION_CAT_FLOW_CONTROL	24	/* 9.6.26 Flow Control */
#define	IEEE80211_ACTION_CAT_CTL_RESP_MCS_NEG	25	/* 9.6.27 Control Response MCS Negotiation */
#define	IEEE80211_ACTION_CAT_FILS		26	/* 9.6.23 FILS */
#define	IEEE80211_ACTION_CAT_CDMG		27	/* 9.6.28 CDMG */
#define	IEEE80211_ACTION_CAT_CMMG		28	/* 9.6.29 CMMG */
#define	IEEE80211_ACTION_CAT_GLK		29	/* 9.6.30 GLK */
#define	IEEE80211_ACTION_CAT_HE			30	/* 9.6.31 HE, 80211ax-2021 */
#define	IEEE80211_ACTION_CAT_PROTECTED_HE	31	/* 9.6.32 Protected HE, 80211ax-2021 */
/* Reserved					32-125 */
#define	IEEE80211_ACTION_CAT_VENDOR_SPECIFIC_PROTECTED 126 /* 9.6.5 Vendor-specific Protected */
#define	IEEE80211_ACTION_CAT_VENDOR		127	/* 9.6.5 Vendor-specific */
/* Error					128-255 */


/* 80211-2020 Table 9-346-Spectrum Management Action field values */
enum ieee80211_action_sm {
	IEEE80211_ACTION_SM_SMREQ		= 0,	/* Spectrum Measurement Request */
	IEEE80211_ACTION_SM_SMREP		= 1,	/* Spectrum Measurement Report */
	IEEE80211_ACTION_SM_TPCREQ		= 2,	/* TPC Request */
	IEEE80211_ACTION_SM_TPCREP		= 3,	/* TPC Report */
	IEEE80211_ACTION_SM_CSA			= 4,	/* Channel Switch Announcement */
	/* Reserved				= 5-255 */
};

/* 80211-2020 Table 9-363-Radio Measurement Action field values */
enum ieee80211_action_radio_measurement {
	IEEE80211_ACTION_RADIO_MEASUREMENT_RMREQ	= 0,	/* Radio Measurement Request */
	IEEE80211_ACTION_RADIO_MEASUREMENT_RMREP	= 1,	/* Radio Measurement Report */
	IEEE80211_ACTION_RADIO_MEASUREMENT_LMREQ	= 2,	/* Link Measurement Request */
	IEEE80211_ACTION_RADIO_MEASUREMENT_LMREP	= 3,	/* Link Measurement Report */
	IEEE80211_ACTION_RADIO_MEASUREMENT_NRREQ	= 4,	/* Neighbor Report Request */
	IEEE80211_ACTION_RADIO_MEASUREMENT_NRRESP	= 5,	/* Neighbor Report Response */
	/* Reserved					= 6-255 */
};

#define	IEEE80211_ACTION_HT_TXCHWIDTH	0	/* recommended xmit chan width*/
#define	IEEE80211_ACTION_HT_MIMOPWRSAVE	1	/* MIMO power save */

/* HT - recommended transmission channel width */
struct ieee80211_action_ht_txchwidth {
	struct ieee80211_action	at_header;
	uint8_t		at_chwidth;	
} __packed;

#define	IEEE80211_A_HT_TXCHWIDTH_20	0
#define	IEEE80211_A_HT_TXCHWIDTH_2040	1

/* HT - MIMO Power Save (NB: D2.04) */
struct ieee80211_action_ht_mimopowersave {
	struct ieee80211_action am_header;
	uint8_t		am_control;
} __packed;

#define	IEEE80211_A_HT_MIMOPWRSAVE_ENA		0x01	/* PS enabled */
#define	IEEE80211_A_HT_MIMOPWRSAVE_MODE		0x02
#define	IEEE80211_A_HT_MIMOPWRSAVE_MODE_S	1
#define	IEEE80211_A_HT_MIMOPWRSAVE_DYNAMIC	0x02	/* Dynamic Mode */
#define	IEEE80211_A_HT_MIMOPWRSAVE_STATIC	0x00	/* no SM packets */
/* bits 2-7 reserved */

/* Block Ack actions */
#define IEEE80211_ACTION_BA_ADDBA_REQUEST       0   /* ADDBA request */
#define IEEE80211_ACTION_BA_ADDBA_RESPONSE      1   /* ADDBA response */
#define IEEE80211_ACTION_BA_DELBA	        2   /* DELBA */

/* Block Ack Parameter Set */
#define	IEEE80211_BAPS_BUFSIZ	0xffc0		/* buffer size */
#define	IEEE80211_BAPS_BUFSIZ_S	6
#define	IEEE80211_BAPS_TID	0x003c		/* TID */
#define	IEEE80211_BAPS_TID_S	2
#define	IEEE80211_BAPS_POLICY	0x0002		/* block ack policy */
#define	IEEE80211_BAPS_POLICY_S	1
#define	IEEE80211_BAPS_AMSDU	0x0001		/* A-MSDU permitted */
#define	IEEE80211_BAPS_AMSDU_S	0

#define	IEEE80211_BAPS_POLICY_DELAYED	(0<<IEEE80211_BAPS_POLICY_S)
#define	IEEE80211_BAPS_POLICY_IMMEDIATE	(1<<IEEE80211_BAPS_POLICY_S)

/* Block Ack Sequence Control */
#define	IEEE80211_BASEQ_START	0xfff0		/* starting seqnum */
#define	IEEE80211_BASEQ_START_S	4
#define	IEEE80211_BASEQ_FRAG	0x000f		/* fragment number */
#define	IEEE80211_BASEQ_FRAG_S	0

/* Delayed Block Ack Parameter Set */
#define	IEEE80211_DELBAPS_TID	0xf000		/* TID */
#define	IEEE80211_DELBAPS_TID_S	12
#define	IEEE80211_DELBAPS_INIT	0x0800		/* initiator */
#define	IEEE80211_DELBAPS_INIT_S 11

/* BA - ADDBA request */
struct ieee80211_action_ba_addbarequest {
	struct ieee80211_action rq_header;
	uint8_t		rq_dialogtoken;
	uint16_t	rq_baparamset;
	uint16_t	rq_batimeout;		/* in TUs */
	uint16_t	rq_baseqctl;
} __packed;

/* BA - ADDBA response */
struct ieee80211_action_ba_addbaresponse {
	struct ieee80211_action rs_header;
	uint8_t		rs_dialogtoken;
	uint16_t	rs_statuscode;
	uint16_t	rs_baparamset; 
	uint16_t	rs_batimeout;		/* in TUs */
} __packed;

/* BA - DELBA */
struct ieee80211_action_ba_delba {
	struct ieee80211_action dl_header;
	uint16_t	dl_baparamset;
	uint16_t	dl_reasoncode;
} __packed;

/* BAR Control */
#define	IEEE80211_BAR_TID	0xf000		/* TID */
#define	IEEE80211_BAR_TID_S	12
#define	IEEE80211_BAR_COMP	0x0004		/* Compressed Bitmap */
#define	IEEE80211_BAR_MTID	0x0002		/* Multi-TID */
#define	IEEE80211_BAR_NOACK	0x0001		/* No-Ack policy */

/* BAR Starting Sequence Control */
#define	IEEE80211_BAR_SEQ_START	0xfff0		/* starting seqnum */
#define	IEEE80211_BAR_SEQ_START_S	4

struct ieee80211_ba_request {
	uint16_t	rq_barctl;
	uint16_t	rq_barseqctl;
} __packed;

/*
 * Control frames.
 */
struct ieee80211_frame_min {
	uint8_t		i_fc[2];
	uint8_t		i_dur[2];
	uint8_t		i_addr1[IEEE80211_ADDR_LEN];
	uint8_t		i_addr2[IEEE80211_ADDR_LEN];
	/* FCS */
} __packed;

struct ieee80211_frame_rts {
	uint8_t		i_fc[2];
	uint8_t		i_dur[2];
	uint8_t		i_ra[IEEE80211_ADDR_LEN];
	uint8_t		i_ta[IEEE80211_ADDR_LEN];
	/* FCS */
} __packed;

struct ieee80211_frame_cts {
	uint8_t		i_fc[2];
	uint8_t		i_dur[2];
	uint8_t		i_ra[IEEE80211_ADDR_LEN];
	/* FCS */
} __packed;

struct ieee80211_frame_ack {
	uint8_t		i_fc[2];
	uint8_t		i_dur[2];
	uint8_t		i_ra[IEEE80211_ADDR_LEN];
	/* FCS */
} __packed;

struct ieee80211_frame_pspoll {
	uint8_t		i_fc[2];
	uint8_t		i_aid[2];
	uint8_t		i_bssid[IEEE80211_ADDR_LEN];
	uint8_t		i_ta[IEEE80211_ADDR_LEN];
	/* FCS */
} __packed;

struct ieee80211_frame_cfend {		/* NB: also CF-End+CF-Ack */
	uint8_t		i_fc[2];
	uint8_t		i_dur[2];	/* should be zero */
	uint8_t		i_ra[IEEE80211_ADDR_LEN];
	uint8_t		i_bssid[IEEE80211_ADDR_LEN];
	/* FCS */
} __packed;

struct ieee80211_frame_bar {
	uint8_t		i_fc[2];
	uint8_t		i_dur[2];
	uint8_t		i_ra[IEEE80211_ADDR_LEN];
	uint8_t		i_ta[IEEE80211_ADDR_LEN];
	uint16_t	i_ctl;
	uint16_t	i_seq;
	/* FCS */
} __packed;

/*
 * BEACON management packets
 *
 *	octet timestamp[8]
 *	octet beacon interval[2]
 *	octet capability information[2]
 *	information element
 *		octet elemid
 *		octet length
 *		octet information[length]
 */

#define	IEEE80211_BEACON_INTERVAL(beacon) \
	((beacon)[8] | ((beacon)[9] << 8))
#define	IEEE80211_BEACON_CAPABILITY(beacon) \
	((beacon)[10] | ((beacon)[11] << 8))

#define	IEEE80211_CAPINFO_ESS			0x0001
#define	IEEE80211_CAPINFO_IBSS			0x0002
#define	IEEE80211_CAPINFO_CF_POLLABLE		0x0004
#define	IEEE80211_CAPINFO_CF_POLLREQ		0x0008
#define	IEEE80211_CAPINFO_PRIVACY		0x0010
#define	IEEE80211_CAPINFO_SHORT_PREAMBLE	0x0020
#define	IEEE80211_CAPINFO_PBCC			0x0040
#define	IEEE80211_CAPINFO_CHNL_AGILITY		0x0080
#define	IEEE80211_CAPINFO_SPECTRUM_MGMT		0x0100
/* bit 9 is reserved */
#define	IEEE80211_CAPINFO_SHORT_SLOTTIME	0x0400
#define	IEEE80211_CAPINFO_RSN			0x0800
/* bit 12 is reserved */
#define	IEEE80211_CAPINFO_DSSSOFDM		0x2000
/* bits 14-15 are reserved */

#define	IEEE80211_CAPINFO_BITS \
	"\20\1ESS\2IBSS\3CF_POLLABLE\4CF_POLLREQ\5PRIVACY\6SHORT_PREAMBLE" \
	"\7PBCC\10CHNL_AGILITY\11SPECTRUM_MGMT\13SHORT_SLOTTIME\14RSN" \
	"\16DSSOFDM"

/*
 * 802.11i/WPA information element (maximally sized).
 */
struct ieee80211_ie_wpa {
	uint8_t		wpa_id;		/* IEEE80211_ELEMID_VENDOR */
	uint8_t		wpa_len;	/* length in bytes */
	uint8_t		wpa_oui[3];	/* 0x00, 0x50, 0xf2 */
	uint8_t		wpa_type;	/* OUI type */
	uint16_t	wpa_version;	/* spec revision */
	uint32_t	wpa_mcipher[1];	/* multicast/group key cipher */
	uint16_t	wpa_uciphercnt;	/* # pairwise key ciphers */
	uint32_t	wpa_uciphers[8];/* ciphers */
	uint16_t	wpa_authselcnt;	/* authentication selector cnt*/
	uint32_t	wpa_authsels[8];/* selectors */
	uint16_t	wpa_caps;	/* 802.11i capabilities */
	uint16_t	wpa_pmkidcnt;	/* 802.11i pmkid count */
	uint16_t	wpa_pmkids[8];	/* 802.11i pmkids */
} __packed;

/*
 * 802.11n HT Capability IE
 * NB: these reflect D1.10 
 */
struct ieee80211_ie_htcap {
	uint8_t		hc_id;			/* element ID */
	uint8_t		hc_len;			/* length in bytes */
	uint16_t	hc_cap;			/* HT caps (see below) */
	uint8_t		hc_param;		/* HT params (see below) */
	uint8_t 	hc_mcsset[16]; 		/* supported MCS set */
	uint16_t	hc_extcap;		/* extended HT capabilities */
	uint32_t	hc_txbf;		/* txbf capabilities */
	uint8_t		hc_antenna;		/* antenna capabilities */
} __packed;

/* HT capability flags (ht_cap) */
#define	IEEE80211_HTCAP_LDPC		0x0001	/* LDPC rx supported */
#define	IEEE80211_HTCAP_CHWIDTH40	0x0002	/* 20/40 supported */
#define	IEEE80211_HTCAP_SMPS		0x000c	/* SM Power Save mode */
#define	IEEE80211_HTCAP_SMPS_OFF	0x000c	/* disabled */
#define	IEEE80211_HTCAP_SMPS_DYNAMIC	0x0004	/* send RTS first */
/* NB: SMPS value 2 is reserved */
#define	IEEE80211_HTCAP_SMPS_ENA	0x0000	/* enabled (static mode) */
#define	IEEE80211_HTCAP_GREENFIELD	0x0010	/* Greenfield supported */
#define	IEEE80211_HTCAP_SHORTGI20	0x0020	/* Short GI in 20MHz */
#define	IEEE80211_HTCAP_SHORTGI40	0x0040	/* Short GI in 40MHz */
#define	IEEE80211_HTCAP_TXSTBC		0x0080	/* STBC tx ok */
#define	IEEE80211_HTCAP_RXSTBC		0x0300  /* STBC rx support */
#define	IEEE80211_HTCAP_RXSTBC_S	8
#define	IEEE80211_HTCAP_RXSTBC_1STREAM	0x0100  /* 1 spatial stream */
#define	IEEE80211_HTCAP_RXSTBC_2STREAM	0x0200  /* 1-2 spatial streams*/
#define	IEEE80211_HTCAP_RXSTBC_3STREAM	0x0300  /* 1-3 spatial streams*/
#define	IEEE80211_HTCAP_DELBA		0x0400	/* HT DELBA supported */
#define	IEEE80211_HTCAP_MAXAMSDU	0x0800	/* max A-MSDU length */
#define	IEEE80211_HTCAP_MAXAMSDU_7935	0x0800	/* 7935 octets */
#define	IEEE80211_HTCAP_MAXAMSDU_3839	0x0000	/* 3839 octets */
#define	IEEE80211_HTCAP_DSSSCCK40	0x1000  /* DSSS/CCK in 40MHz */
#define	IEEE80211_HTCAP_PSMP		0x2000  /* PSMP supported */
#define	IEEE80211_HTCAP_40INTOLERANT	0x4000  /* 40MHz intolerant */
#define	IEEE80211_HTCAP_LSIGTXOPPROT	0x8000  /* L-SIG TXOP prot */

#define	IEEE80211_HTCAP_BITS \
	"\20\1LDPC\2CHWIDTH40\5GREENFIELD\6SHORTGI20\7SHORTGI40\10TXSTBC" \
	"\13DELBA\14AMSDU(7935)\15DSSSCCK40\16PSMP\1740INTOLERANT" \
	"\20LSIGTXOPPROT"

/* HT parameters (hc_param) */
#define	IEEE80211_HTCAP_MAXRXAMPDU	0x03	/* max rx A-MPDU factor */
#define	IEEE80211_HTCAP_MAXRXAMPDU_S	0
#define	IEEE80211_HTCAP_MAXRXAMPDU_8K	0
#define	IEEE80211_HTCAP_MAXRXAMPDU_16K	1
#define	IEEE80211_HTCAP_MAXRXAMPDU_32K	2
#define	IEEE80211_HTCAP_MAXRXAMPDU_64K	3
#define	IEEE80211_HTCAP_MPDUDENSITY	0x1c	/* min MPDU start spacing */
#define	IEEE80211_HTCAP_MPDUDENSITY_S	2
#define	IEEE80211_HTCAP_MPDUDENSITY_NA	0	/* no time restriction */
#define	IEEE80211_HTCAP_MPDUDENSITY_025	1	/* 1/4 us */
#define	IEEE80211_HTCAP_MPDUDENSITY_05	2	/* 1/2 us */
#define	IEEE80211_HTCAP_MPDUDENSITY_1	3	/* 1 us */
#define	IEEE80211_HTCAP_MPDUDENSITY_2	4	/* 2 us */
#define	IEEE80211_HTCAP_MPDUDENSITY_4	5	/* 4 us */
#define	IEEE80211_HTCAP_MPDUDENSITY_8	6	/* 8 us */
#define	IEEE80211_HTCAP_MPDUDENSITY_16	7	/* 16 us */

/* HT extended capabilities (hc_extcap) */
#define	IEEE80211_HTCAP_PCO		0x0001	/* PCO capable */
#define	IEEE80211_HTCAP_PCOTRANS	0x0006	/* PCO transition time */
#define	IEEE80211_HTCAP_PCOTRANS_S	1
#define	IEEE80211_HTCAP_PCOTRANS_04	0x0002	/* 400 us */
#define	IEEE80211_HTCAP_PCOTRANS_15	0x0004	/* 1.5 ms */
#define	IEEE80211_HTCAP_PCOTRANS_5	0x0006	/* 5 ms */
/* bits 3-7 reserved */
#define	IEEE80211_HTCAP_MCSFBACK	0x0300	/* MCS feedback */
#define	IEEE80211_HTCAP_MCSFBACK_S	8
#define	IEEE80211_HTCAP_MCSFBACK_NONE	0x0000	/* nothing provided */
#define	IEEE80211_HTCAP_MCSFBACK_UNSOL	0x0200	/* unsolicited feedback */
#define	IEEE80211_HTCAP_MCSFBACK_MRQ	0x0300	/* " "+respond to MRQ */
#define	IEEE80211_HTCAP_HTC		0x0400	/* +HTC support */
#define	IEEE80211_HTCAP_RDR		0x0800	/* reverse direction responder*/
/* bits 12-15 reserved */

/*
 * 802.11n HT Information IE
 */
struct ieee80211_ie_htinfo {
	uint8_t		hi_id;			/* element ID */
	uint8_t		hi_len;			/* length in bytes */
	uint8_t		hi_ctrlchannel;		/* primary channel */
	uint8_t		hi_byte1;		/* ht ie byte 1 */
	uint8_t		hi_byte2;		/* ht ie byte 2 */
	uint8_t		hi_byte3;		/* ht ie byte 3 */
	uint16_t	hi_byte45;		/* ht ie bytes 4+5 */
	uint8_t 	hi_basicmcsset[16]; 	/* basic MCS set */
} __packed;

/* byte1 */
#define	IEEE80211_HTINFO_2NDCHAN	0x03	/* secondary/ext chan offset */
#define	IEEE80211_HTINFO_2NDCHAN_S	0
#define	IEEE80211_HTINFO_2NDCHAN_NONE	0x00	/* no secondary/ext channel */
#define	IEEE80211_HTINFO_2NDCHAN_ABOVE	0x01	/* above private channel */
/* NB: 2 is reserved */
#define	IEEE80211_HTINFO_2NDCHAN_BELOW	0x03	/* below primary channel */ 
#define	IEEE80211_HTINFO_TXWIDTH	0x04	/* tx channel width */
#define	IEEE80211_HTINFO_TXWIDTH_20	0x00	/* 20MHz width */
#define	IEEE80211_HTINFO_TXWIDTH_2040	0x04	/* any supported width */
#define	IEEE80211_HTINFO_RIFSMODE	0x08	/* Reduced IFS (RIFS) use */
#define	IEEE80211_HTINFO_RIFSMODE_PROH	0x00	/* RIFS use prohibited */
#define	IEEE80211_HTINFO_RIFSMODE_PERM	0x08	/* RIFS use permitted */
#define	IEEE80211_HTINFO_PMSPONLY	0x10	/* PSMP required to associate */
#define	IEEE80211_HTINFO_SIGRAN		0xe0	/* shortest Service Interval */
#define	IEEE80211_HTINFO_SIGRAN_S	5
#define	IEEE80211_HTINFO_SIGRAN_5	0x00	/* 5 ms */
/* XXX add rest */

/* bytes 2+3 */
#define	IEEE80211_HTINFO_OPMODE		0x03	/* operating mode */
#define	IEEE80211_HTINFO_OPMODE_S	0
#define	IEEE80211_HTINFO_OPMODE_PURE	0x00	/* no protection */
#define	IEEE80211_HTINFO_OPMODE_PROTOPT	0x01	/* protection optional */
#define	IEEE80211_HTINFO_OPMODE_HT20PR	0x02	/* protection for HT20 sta's */
#define	IEEE80211_HTINFO_OPMODE_MIXED	0x03	/* protection for legacy sta's*/
#define	IEEE80211_HTINFO_NONGF_PRESENT	0x04	/* non-GF sta's present */
#define	IEEE80211_HTINFO_TXBL		0x08	/* transmit burst limit */
#define	IEEE80211_HTINFO_NONHT_PRESENT	0x10	/* non-HT sta's present */
/* bits 5-15 reserved */

/* bytes 4+5 */
#define	IEEE80211_HTINFO_2NDARYBEACON	0x01
#define	IEEE80211_HTINFO_LSIGTXOPPROT	0x02
#define	IEEE80211_HTINFO_PCO_ACTIVE	0x04
#define	IEEE80211_HTINFO_40MHZPHASE	0x08

/* byte5 */
#define	IEEE80211_HTINFO_BASIC_STBCMCS	0x7f
#define	IEEE80211_HTINFO_BASIC_STBCMCS_S 0
#define	IEEE80211_HTINFO_DUALPROTECTED	0x80

/*
 * 802.11ac definitions - 802.11ac-2013 .
 */

/*
 * Maximum length of A-MPDU that the STA can RX in VHT.
 * Length = 2 ^ (13 + max_ampdu_length_exp) - 1 (octets)
 */
#define	IEEE80211_VHTCAP_MAX_AMPDU_8K		0
#define	IEEE80211_VHTCAP_MAX_AMPDU_16K		1
#define	IEEE80211_VHTCAP_MAX_AMPDU_32K		2
#define	IEEE80211_VHTCAP_MAX_AMPDU_64K		3
#define	IEEE80211_VHTCAP_MAX_AMPDU_128K		4
#define	IEEE80211_VHTCAP_MAX_AMPDU_256K		5
#define	IEEE80211_VHTCAP_MAX_AMPDU_512K		6
#define	IEEE80211_VHTCAP_MAX_AMPDU_1024K	7

/*
 * VHT MCS information.
 * + rx_highest/tx_highest: optional; maximum long GI VHT PPDU
 *    data rate.  1Mbit/sec units.
 * + rx_mcs_map/tx_mcs_map: bitmap of per-stream supported MCS;
 *    2 bits each.
 */

/* 802.11ac-2013, 8.4.2.160.3 Supported VHT-MCS and NSS Set field */
enum ieee80211_vht_mcs_support {
	IEEE80211_VHT_MCS_SUPPORT_0_7		= 0,	/* MCS0-7 */
	IEEE80211_VHT_MCS_SUPPORT_0_8		= 1,	/* MCS0-8 */
	IEEE80211_VHT_MCS_SUPPORT_0_9		= 2,	/* MCS0-9 */
	IEEE80211_VHT_MCS_NOT_SUPPORTED		= 3	/* not supported */
};

/* 802.11ac-2013, 8.4.2.160.3 Supported VHT-MCS and NSS Set field */
struct ieee80211_vht_mcs_info {
	uint16_t rx_mcs_map;
	uint16_t rx_highest;
	uint16_t tx_mcs_map;
	uint16_t tx_highest;
} __packed;

/* 802.11ac-2013, 8.4.2.160.1 VHT Capabilities element structure */
struct ieee80211_vht_cap {
	uint32_t			vht_cap_info;
	struct ieee80211_vht_mcs_info	supp_mcs;
} __packed;

/*
 * 802.11ac-2013, Table 8-183x-VHT Operation Information subfields.
 * 802.11-2020, Table 9-274-VHT Operation Information subfields (for deprecations)
 */
enum ieee80211_vht_chanwidth {
	IEEE80211_VHT_CHANWIDTH_USE_HT		= 0,	/* 20 MHz or 40 MHz */
	IEEE80211_VHT_CHANWIDTH_80MHZ		= 1,	/* 80MHz */
	IEEE80211_VHT_CHANWIDTH_160MHZ		= 2,	/* 160MHz (deprecated) */
	IEEE80211_VHT_CHANWIDTH_80P80MHZ	= 3,	/* 80+80MHz (deprecated) */
	/* 4..255 reserved. */
};

/* The name conflicts with the same structure in wpa.  Only ifconfig needs this. */
#if defined(_KERNEL) || defined(WANT_NET80211)
/* 802.11ac-2013 8.4.2.161 VHT Operation element */
struct ieee80211_vht_operation {
	uint8_t			chan_width;		/* enum ieee80211_vht_chanwidth */
	uint8_t			center_freq_seq0_idx;	/* 20/40/80/160 - VHT chan1 */
	uint8_t			center_freq_seq1_idx;	/* 80+80 - VHT chan2 */
	uint16_t		basic_mcs_set;		/* Basic VHT-MCS and NSS Set */
} __packed;
#endif

/* 802.11ac VHT Capabilities */
#define	IEEE80211_VHTCAP_MAX_MPDU_LENGTH_3895	0x00000000
#define	IEEE80211_VHTCAP_MAX_MPDU_LENGTH_7991	0x00000001
#define	IEEE80211_VHTCAP_MAX_MPDU_LENGTH_11454	0x00000002
#define	IEEE80211_VHTCAP_MAX_MPDU_MASK		0x00000003
#define	IEEE80211_VHTCAP_MAX_MPDU_MASK_S	0

#define	IEEE80211_VHTCAP_SUPP_CHAN_WIDTH_MASK	0x0000000C
#define	IEEE80211_VHTCAP_SUPP_CHAN_WIDTH_MASK_S	2
#define	IEEE80211_VHTCAP_SUPP_CHAN_WIDTH_NO160		0
#define	IEEE80211_VHTCAP_SUPP_CHAN_WIDTH_160MHZ		1
#define	IEEE80211_VHTCAP_SUPP_CHAN_WIDTH_160_80P80MHZ	2
#define	IEEE80211_VHTCAP_SUPP_CHAN_WIDTH_RESERVED	3

#define	IEEE80211_VHTCAP_SUPP_CHAN_WIDTH_IS_160MHZ(_vhtcaps)		\
    (_IEEE80211_MASKSHIFT(_vhtcaps, IEEE80211_VHTCAP_SUPP_CHAN_WIDTH_MASK) >= \
            IEEE80211_VHTCAP_SUPP_CHAN_WIDTH_160MHZ)
#define	IEEE80211_VHTCAP_SUPP_CHAN_WIDTH_IS_160_80P80MHZ(_vhtcaps)	\
    (_IEEE80211_MASKSHIFT(_vhtcaps, IEEE80211_VHTCAP_SUPP_CHAN_WIDTH_MASK) == \
            IEEE80211_VHTCAP_SUPP_CHAN_WIDTH_160_80P80MHZ)

#define	IEEE80211_VHTCAP_RXLDPC		0x00000010
#define	IEEE80211_VHTCAP_RXLDPC_S	4

#define	IEEE80211_VHTCAP_SHORT_GI_80		0x00000020
#define	IEEE80211_VHTCAP_SHORT_GI_80_S		5

#define	IEEE80211_VHTCAP_SHORT_GI_160		0x00000040
#define	IEEE80211_VHTCAP_SHORT_GI_160_S		6

#define	IEEE80211_VHTCAP_TXSTBC		0x00000080
#define	IEEE80211_VHTCAP_TXSTBC_S	7

#define	IEEE80211_VHTCAP_RXSTBC_1		0x00000100
#define	IEEE80211_VHTCAP_RXSTBC_2		0x00000200
#define	IEEE80211_VHTCAP_RXSTBC_3		0x00000300
#define	IEEE80211_VHTCAP_RXSTBC_4		0x00000400
#define	IEEE80211_VHTCAP_RXSTBC_MASK		0x00000700
#define	IEEE80211_VHTCAP_RXSTBC_MASK_S		8

#define	IEEE80211_VHTCAP_SU_BEAMFORMER_CAPABLE	0x00000800
#define	IEEE80211_VHTCAP_SU_BEAMFORMER_CAPABLE_S	11

#define	IEEE80211_VHTCAP_SU_BEAMFORMEE_CAPABLE	0x00001000
#define	IEEE80211_VHTCAP_SU_BEAMFORMEE_CAPABLE_S	12

#define	IEEE80211_VHTCAP_BEAMFORMEE_STS_SHIFT	13
#define	IEEE80211_VHTCAP_BEAMFORMEE_STS_MASK \
	    (7 << IEEE80211_VHTCAP_BEAMFORMEE_STS_SHIFT)
#define	IEEE80211_VHTCAP_BEAMFORMEE_STS_MASK_S	13

#define	IEEE80211_VHTCAP_SOUNDING_DIMENSIONS_SHIFT	16
#define	IEEE80211_VHTCAP_SOUNDING_DIMENSIONS_MASK \
	    (7 << IEEE80211_VHTCAP_SOUNDING_DIMENSIONS_SHIFT)
#define	IEEE80211_VHTCAP_SOUNDING_DIMENSIONS_MASK_S	16

#define	IEEE80211_VHTCAP_MU_BEAMFORMER_CAPABLE	0x00080000
#define	IEEE80211_VHTCAP_MU_BEAMFORMER_CAPABLE_S	19
#define	IEEE80211_VHTCAP_MU_BEAMFORMEE_CAPABLE	0x00100000
#define	IEEE80211_VHTCAP_MU_BEAMFORMEE_CAPABLE_S	20
#define	IEEE80211_VHTCAP_VHT_TXOP_PS		0x00200000
#define	IEEE80211_VHTCAP_VHT_TXOP_PS_S		21
#define	IEEE80211_VHTCAP_HTC_VHT		0x00400000
#define	IEEE80211_VHTCAP_HTC_VHT_S		22

#define	IEEE80211_VHTCAP_MAX_A_MPDU_LENGTH_EXPONENT_SHIFT	23
#define	IEEE80211_VHTCAP_MAX_A_MPDU_LENGTH_EXPONENT_MASK \
	    (7 << IEEE80211_VHTCAP_MAX_A_MPDU_LENGTH_EXPONENT_SHIFT)
#define	IEEE80211_VHTCAP_MAX_A_MPDU_LENGTH_EXPONENT_MASK_S	23

#define	IEEE80211_VHTCAP_VHT_LINK_ADAPTATION_VHT_MASK	0x0c000000
#define	IEEE80211_VHTCAP_VHT_LINK_ADAPTATION_VHT_UNSOL_MFB	0x08000000
#define	IEEE80211_VHTCAP_VHT_LINK_ADAPTATION_VHT_MRQ_MFB	0x0c000000
#define	IEEE80211_VHTCAP_VHT_LINK_ADAPTATION_VHT_MASK_S	26

#define	IEEE80211_VHTCAP_RX_ANTENNA_PATTERN	0x10000000
#define	IEEE80211_VHTCAP_RX_ANTENNA_PATTERN_S	28
#define	IEEE80211_VHTCAP_TX_ANTENNA_PATTERN	0x20000000
#define	IEEE80211_VHTCAP_TX_ANTENNA_PATTERN_S	29

/* 802.11-2016, 9.4.2.158.2 VHT Capabilities Information field. */
#define	IEEE80211_VHTCAP_EXT_NSS_BW		0xc0000000
#define	IEEE80211_VHTCAP_EXT_NSS_BW_S		30

/*
 * XXX TODO: add the rest of the bits
 */
#define	IEEE80211_VHTCAP_BITS \
	"\20\1MPDU7991\2MPDU11454\3CHAN160\4CHAN160+80P80\5RXLDPC\6SHORTGI80" \
	"\7SHORTGI160\10RXSTBC1\11RXSTBC2\12RXSTBC3\13RXSTBC4\14BFERCAP" \
	"\15BFEECAP\27VHT\37RXANTPTN\40TXANTPTN"

/*
 * VHT Transmit Power Envelope element - 802.11ac-2013 8.4.2.164
 *
 * This defines the maximum transmit power for various bandwidths.
 */
/*
 * Count is how many elements follow and what they're for:
 *
 * 0 - 20 MHz
 * 1 - 20+40 MHz
 * 2 - 20+40+80 MHz
 * 3 - 20+40+80+(160, 80+80) MHz
 */
#define	IEEE80211_VHT_TXPWRENV_INFO_COUNT_SHIFT	0
#define	IEEE80211_VHT_TXPWRENV_INFO_COUNT_MASK	0x07

/*
 * Unit is the tx power representation.  It should be EIRP for now;
 * other values are reserved.
 */
#define	IEEE80211_VHT_TXPWRENV_UNIT_MASK	0x38
#define	IEEE80211_VHT_TXPWRENV_UNIT_SHIFT	3

/* This value is within the unit mask/shift above */
#define	IEEE80211_VHT_TXPWRENV_UNIT_EIRP	0

struct ieee80211_ie_vht_txpwrenv {
	uint8_t ie;
	uint8_t len;
	uint8_t tx_info;
	int8_t tx_elem[0];	/* TX power elements, 1/2 dB, signed */
};

/* VHT action codes */
#define	WLAN_ACTION_VHT_COMPRESSED_BF		0
#define	WLAN_ACTION_VHT_GROUPID_MGMT		1
#define	WLAN_ACTION_VHT_OPMODE_NOTIF		2

#if defined(_KERNEL) || defined(WANT_NET80211)
/*
 * HE
 */

/*
 * 802.11ax-2021, 9.4.2.248.2 HE MAC Capabilities Information field.
 */
/* B0..B7 */
#define	IEEE80211_HE_MAC_CAP0_HTC_HE			(1<<0)
#define	IEEE80211_HE_MAC_CAP0_TWT_REQ			(1<<1)
#define	IEEE80211_HE_MAC_CAP0_TWT_RES			(1<<2)

/* B8..B15 */
#define	IEEE80211_HE_MAC_CAP1_TF_MAC_PAD_DUR_16US	(1<<3)
#define	IEEE80211_HE_MAC_CAP1_TF_MAC_PAD_DUR_MASK	(1<<3 | 1<<2)
#define	IEEE80211_HE_MAC_CAP1_MULTI_TID_AGG_RX_QOS_8	(1<<6 | 1<<5 | 1<<4)
#define	IEEE80211_HE_MAC_CAP1_LINK_ADAPTATION		(1<<7)
/* Note: B15|B16 are split between octets %!$@??? */

/* B16..B23 */
#define	IEEE80211_HE_MAC_CAP2_LINK_ADAPTATION		(1<<0)
#define	IEEE80211_HE_MAC_CAP2_ALL_ACK			(1<<1)
#define	IEEE80211_HE_MAC_CAP2_TRS			(1<<2)
#define	IEEE80211_HE_MAC_CAP2_BSR			(1<<3)
#define	IEEE80211_HE_MAC_CAP2_BCAST_TWT			(1<<4)
#define	IEEE80211_HE_MAC_CAP2_32BIT_BA_BITMAP		(1<<5)
#define	IEEE80211_HE_MAC_CAP2_MU_CASCADING		(1<<6)
#define	IEEE80211_HE_MAC_CAP2_ACK_EN			(1<<7)

/* B24..B31 */
#define	IEEE80211_HE_MAC_CAP3_OMI_CONTROL		(1<<1)
#define	IEEE80211_HE_MAC_CAP3_OFDMA_RA			(1<<2)
#define	IEEE80211_HE_MAC_CAP3_MAX_AMPDU_LEN_EXP_EXT_1	(1<<3)
#define	IEEE80211_HE_MAC_CAP3_MAX_AMPDU_LEN_EXP_EXT_2	(1<<4)
#define	IEEE80211_HE_MAC_CAP3_MAX_AMPDU_LEN_EXP_EXT_3	(1<<4 | 1<<3)
#define	IEEE80211_HE_MAC_CAP3_MAX_AMPDU_LEN_EXP_MASK	(1<<4 | 1<<3)
#define	IEEE80211_HE_MAC_CAP3_FLEX_TWT_SCHED		(1<<6)
#define	IEEE80211_HE_MAC_CAP3_RX_CTRL_FRAME_TO_MULTIBSS	(1<<7)

/* B32..B39 */
#define	IEEE80211_HE_MAC_CAP4_BSRP_BQRP_A_MPDU_AGG	(1<<0)
#define	IEEE80211_HE_MAC_CAP4_BQR			(1<<2)
#define	IEEE80211_HE_MAC_CAP4_OPS			(1<<5)
#define	IEEE80211_HE_MAC_CAP4_AMSDU_IN_AMPDU		(1<<6)
#define	IEEE80211_HE_MAC_CAP4_MULTI_TID_AGG_TX_QOS_B39	(1<<7)
/* Note: B39|B40|B41 are split between octets %!$@??? */

/* B40..B47 */
#define	IEEE80211_HE_MAC_CAP5_MULTI_TID_AGG_TX_QOS_B40	(1<<0)
#define	IEEE80211_HE_MAC_CAP5_MULTI_TID_AGG_TX_QOS_B41	(1<<1)
#define	IEEE80211_HE_MAC_CAP5_SUBCHAN_SELECTIVE_TRANSMISSION	(1<<2)
#define	IEEE80211_HE_MAC_CAP5_UL_2x996_TONE_RU		(1<<3)
#define	IEEE80211_HE_MAC_CAP5_OM_CTRL_UL_MU_DATA_DIS_RX	(1<<4)
#define	IEEE80211_HE_MAC_CAP5_HE_DYNAMIC_SM_PS		(1<<5)
#define	IEEE80211_HE_MAC_CAP5_PUNCTURED_SOUNDING	(1<<6)
#define	IEEE80211_HE_MAC_CAP5_HT_VHT_TRIG_FRAME_RX	(1<<7)

/*
 * 802.11ax-2021, 9.4.2.248.3 HE PHY Capabilities Information field.
 */
/* B0..B7 */
#define	IEEE80211_HE_PHY_CAP0_CHANNEL_WIDTH_SET_40MHZ_IN_2G		(1<<1)
#define	IEEE80211_HE_PHY_CAP0_CHANNEL_WIDTH_SET_40MHZ_80MHZ_IN_5G	(1<<2)
#define	IEEE80211_HE_PHY_CAP0_CHANNEL_WIDTH_SET_160MHZ_IN_5G		(1<<3)
#define	IEEE80211_HE_PHY_CAP0_CHANNEL_WIDTH_SET_80PLUS80_MHZ_IN_5G	(1<<4)
#define	IEEE80211_HE_PHY_CAP0_CHANNEL_WIDTH_SET_MASK_ALL		(1<<4 | 1<<3 | 1<<2 | 1<<1)
#define	IEEE80211_HE_PHY_CAP0_CHANNEL_WIDTH_SET_RU_MAPPING_IN_2G	(1<<5)
#define	IEEE80211_HE_PHY_CAP0_CHANNEL_WIDTH_SET_RU_MAPPING_IN_5G	(1<<6)
#define	IEEE80211_HE_PHY_CAP0_CHANNEL_WIDTH_SET_MASK			(1<<6 | 1<<5 | 1<<4 | 1<<3 | 1<<2 | 1<<1)

/* B8..B15 */
#define	IEEE80211_HE_PHY_CAP1_PREAMBLE_PUNC_RX_MASK	(1<<3 | 1<<2 | 1<<1 | 1<<0)
#define	IEEE80211_HE_PHY_CAP1_DEVICE_CLASS_A		(1<<4)
#define	IEEE80211_HE_PHY_CAP1_LDPC_CODING_IN_PAYLOAD	(1<<5)
#define	IEEE80211_HE_PHY_CAP1_HE_LTF_AND_GI_FOR_HE_PPDUS_0_8US	(1<<6)
#define	IEEE80211_HE_PHY_CAP1_MIDAMBLE_RX_TX_MAX_NSTS	(1<<7)
/* Note: B15|B16 are split between octets %!$@??? */

/* B16..B23 */
#define	IEEE80211_HE_PHY_CAP2_MIDAMBLE_RX_TX_MAX_NSTS	(1<<0)
#define	IEEE80211_HE_PHY_CAP2_NDP_4x_LTF_AND_3_2US	(1<<1)
#define	IEEE80211_HE_PHY_CAP2_STBC_TX_UNDER_80MHZ	(1<<2)
#define	IEEE80211_HE_PHY_CAP2_STBC_RX_UNDER_80MHZ	(1<<3)
#define	IEEE80211_HE_PHY_CAP2_DOPPLER_TX		(1<<4)
#define	IEEE80211_HE_PHY_CAP2_DOPPLER_RX		(1<<5)
#define	IEEE80211_HE_PHY_CAP2_UL_MU_FULL_MU_MIMO	(1<<6)
#define	IEEE80211_HE_PHY_CAP2_UL_MU_PARTIAL_MU_MIMO	(1<<7)

/* B24..B31 */
#define	IEEE80211_HE_PHY_CAP3_DCM_MAX_CONST_TX_NO_DCM	0x00
#define	IEEE80211_HE_PHY_CAP3_DCM_MAX_CONST_TX_BPSK	(1<<0)
#define	IEEE80211_HE_PHY_CAP3_DCM_MAX_CONST_TX_QPSK	(1<<1)
#define	IEEE80211_HE_PHY_CAP3_DCM_MAX_CONST_TX_16_QAM	(1<<1 | 1<<0)
#define	IEEE80211_HE_PHY_CAP3_DCM_MAX_CONST_TX_MASK	(1<<1 | 1<<0)
#define	IEEE80211_HE_PHY_CAP3_DCM_MAX_TX_NSS_1		0x00
#define	IEEE80211_HE_PHY_CAP3_DCM_MAX_TX_NSS_2		(1<<2)
#define	IEEE80211_HE_PHY_CAP3_DCM_MAX_CONST_RX_NO_DCM	0x00
#define	IEEE80211_HE_PHY_CAP3_DCM_MAX_CONST_RX_BPSK	(1<<3)
#define	IEEE80211_HE_PHY_CAP3_DCM_MAX_CONST_RX_QPSK	(1<<4)
#define	IEEE80211_HE_PHY_CAP3_DCM_MAX_CONST_RX_16_QAM	(1<<4 | 1<<3)
#define	IEEE80211_HE_PHY_CAP3_DCM_MAX_CONST_RX_MASK	(1<<4 | 1<<3)
#define	IEEE80211_HE_PHY_CAP3_DCM_MAX_RX_NSS_1		0x00
#define	IEEE80211_HE_PHY_CAP3_DCM_MAX_RX_NSS_2		(1<<5)
#define	IEEE80211_HE_PHY_CAP3_RX_PARTIAL_BW_SU_IN_20MHZ_MU	(1<<6)
#define	IEEE80211_HE_PHY_CAP3_SU_BEAMFORMER		(1<<7)

/* B32..B39 */
#define	IEEE80211_HE_PHY_CAP4_SU_BEAMFORMEE			(1<<0)
#define	IEEE80211_HE_PHY_CAP4_MU_BEAMFORMER			(1<<1)
#define	IEEE80211_HE_PHY_CAP4_BEAMFORMEE_MAX_STS_UNDER_80MHZ_4	(1<<3 | 1<<2)
#define	IEEE80211_HE_PHY_CAP4_BEAMFORMEE_MAX_STS_UNDER_80MHZ_5	(1<<4)
#define	IEEE80211_HE_PHY_CAP4_BEAMFORMEE_MAX_STS_UNDER_80MHZ_8	(1<<4 | 1<<3 | 1<<2)
#define	IEEE80211_HE_PHY_CAP4_BEAMFORMEE_MAX_STS_UNDER_80MHZ_MASK	(1<<4 | 1<<3 | 1<<2)
#define	IEEE80211_HE_PHY_CAP4_BEAMFORMEE_MAX_STS_ABOVE_80MHZ_4	(1<<6 | 1<<5)
#define	IEEE80211_HE_PHY_CAP4_BEAMFORMEE_MAX_STS_ABOVE_80MHZ_5	(1<<7)
#define	IEEE80211_HE_PHY_CAP4_BEAMFORMEE_MAX_STS_ABOVE_80MHZ_8	(1<<7 | 1<<6 | 1<<5)
#define	IEEE80211_HE_PHY_CAP4_BEAMFORMEE_MAX_STS_ABOVE_80MHZ_MASK	(1<<7 | 1<<6 | 1<<5)

/* B40..B47 */
#define	IEEE80211_HE_PHY_CAP5_BEAMFORMEE_NUM_SND_DIM_UNDER_80MHZ_2	(1<<0)
#define	IEEE80211_HE_PHY_CAP5_BEAMFORMEE_NUM_SND_DIM_UNDER_80MHZ_MASK	(1<<2 | 1<<1 | 1<<0)
#define	IEEE80211_HE_PHY_CAP5_BEAMFORMEE_NUM_SND_DIM_ABOVE_80MHZ_2	(1<<3)
#define	IEEE80211_HE_PHY_CAP5_BEAMFORMEE_NUM_SND_DIM_ABOVE_80MHZ_MASK	(1<<5 | 1<<4 | 1<<3)
#define	IEEE80211_HE_PHY_CAP5_NG16_SU_FEEDBACK			(1<<6)
#define	IEEE80211_HE_PHY_CAP5_NG16_MU_FEEDBACK			(1<<7)

/* B48..B55 */
#define	IEEE80211_HE_PHY_CAP6_CODEBOOK_SIZE_42_SU	(1<<0)
#define	IEEE80211_HE_PHY_CAP6_CODEBOOK_SIZE_75_MU	(1<<1)
#define	IEEE80211_HE_PHY_CAP6_TRIG_SU_BEAMFORMING_FB	(1<<2)
#define	IEEE80211_HE_PHY_CAP6_TRIG_MU_BEAMFORMING_PARTIAL_BW_FB	(1<<3)
#define	IEEE80211_HE_PHY_CAP6_TRIG_CQI_FB		(1<<4)
#define	IEEE80211_HE_PHY_CAP6_PARTIAL_BW_EXT_RANGE	(1<<5)
#define	IEEE80211_HE_PHY_CAP6_PARTIAL_BANDWIDTH_DL_MUMIMO	(1<<6)
#define	IEEE80211_HE_PHY_CAP6_PPE_THRESHOLD_PRESENT	(1<<7)

/* B56..B63 */
#define	IEEE80211_HE_PHY_CAP7_PSR_BASED_SR			(1<<0)
#define	IEEE80211_HE_PHY_CAP7_POWER_BOOST_FACTOR_SUPP		(1<<1)
#define	IEEE80211_HE_PHY_CAP7_HE_SU_MU_PPDU_4XLTF_AND_08_US_GI	(1<<2)
#define	IEEE80211_HE_PHY_CAP7_MAX_NC_1				(1<<3)
#define	IEEE80211_HE_PHY_CAP7_MAX_NC_2				(1<<4)
#define	IEEE80211_HE_PHY_CAP7_MAX_NC_MASK			(1<<5 | 1<<4 | 1<<3)
#define	IEEE80211_HE_PHY_CAP7_STBC_TX_ABOVE_80MHZ		(1<<6)
#define	IEEE80211_HE_PHY_CAP7_STBC_RX_ABOVE_80MHZ		(1<<7)

/* B64..B71 */
#define	IEEE80211_HE_PHY_CAP8_HE_ER_SU_PPDU_4XLTF_AND_08_US_GI	(1<<0)
#define	IEEE80211_HE_PHY_CAP8_20MHZ_IN_40MHZ_HE_PPDU_IN_2G	(1<<1)
#define	IEEE80211_HE_PHY_CAP8_20MHZ_IN_160MHZ_HE_PPDU		(1<<2)
#define	IEEE80211_HE_PHY_CAP8_80MHZ_IN_160MHZ_HE_PPDU		(1<<3)
#define	IEEE80211_HE_PHY_CAP8_HE_ER_SU_1XLTF_AND_08_US_GI	(1<<4)
#define	IEEE80211_HE_PHY_CAP8_DCM_MAX_RU_242			0x00
#define	IEEE80211_HE_PHY_CAP8_DCM_MAX_RU_484			(1<<6)
#define	IEEE80211_HE_PHY_CAP8_DCM_MAX_RU_996			(1<<7)
#define	IEEE80211_HE_PHY_CAP8_DCM_MAX_RU_2x996			(1<<7 | 1<<6)
#define	IEEE80211_HE_PHY_CAP8_DCM_MAX_RU_MASK			(1<<7 | 1<<6)

/* B72..B79 */
#define	IEEE80211_HE_PHY_CAP9_LONGER_THAN_16_SIGB_OFDM_SYM	(1<<0)
#define	IEEE80211_HE_PHY_CAP9_NON_TRIGGERED_CQI_FEEDBACK	(1<<1)
#define	IEEE80211_HE_PHY_CAP9_TX_1024_QAM_LESS_THAN_242_TONE_RU	(1<<2)
#define	IEEE80211_HE_PHY_CAP9_RX_1024_QAM_LESS_THAN_242_TONE_RU	(1<<3)
#define	IEEE80211_HE_PHY_CAP9_RX_FULL_BW_SU_USING_MU_WITH_COMP_SIGB	(1<<4)
#define	IEEE80211_HE_PHY_CAP9_RX_FULL_BW_SU_USING_MU_WITH_NON_COMP_SIGB	(1<<5)
#define	IEEE80211_HE_PHY_CAP9_NOMINAL_PKT_PADDING_0US		0x00
#define	IEEE80211_HE_PHY_CAP9_NOMINAL_PKT_PADDING_8US		1
#define	IEEE80211_HE_PHY_CAP9_NOMINAL_PKT_PADDING_16US		2
#define	IEEE80211_HE_PHY_CAP9_NOMINAL_PKT_PADDING_RESERVED	3
#define	IEEE80211_HE_PHY_CAP9_NOMINAL_PKT_PADDING_POS		6
#define	IEEE80211_HE_PHY_CAP9_NOMINAL_PKT_PADDING_MASK		(1<<7 | 1<<6)

/* B80..B87 */
#define	IEEE80211_HE_PHY_CAP10_HE_MU_M1RU_MAX_LTF		(1<<0)

/*
 * 802.11ax-2021,
 * 9.4.2.248.2 HE MAC Capabilities Information field.
 * 9.4.2.248.3 HE PHY Capabilities Information field.
 */
struct ieee80211_he_cap_elem {
	uint8_t					mac_cap_info[6];
	uint8_t					phy_cap_info[11];
} __packed;

/* 802.11ax-2021, 9.4.2.248.4 Supported HE-MCS And NSS Set field. */
enum ieee80211_he_mcs_support {
	IEEE80211_HE_MCS_SUPPORT_0_7	= 0,	/* HE-MCS 0-7 for n NSS */
	IEEE80211_HE_MCS_SUPPORT_0_9	= 1,	/* HE-MCS 0-9 for n NSS */
	IEEE80211_HE_MCS_SUPPORT_0_11	= 2,	/* HE-MCS 0-11 for n NSS */
	IEEE80211_HE_MCS_NOT_SUPPORTED	= 3	/* n NSS not supported. */
};

struct ieee80211_he_mcs_nss_supp {
	uint16_t				rx_mcs_80;
	uint16_t				tx_mcs_80;
	uint16_t				rx_mcs_160;
	uint16_t				tx_mcs_160;
	uint16_t				rx_mcs_80p80;
	uint16_t				tx_mcs_80p80;
} __packed;

#define	IEEE80211_HE_CAP_PPE_THRES_MAX		25

/* XXX this should only be internal. */
struct net80211_he_cap {
	bool					has_he;
	struct ieee80211_he_cap_elem		he_cap_elem;
	struct ieee80211_he_mcs_nss_supp	he_mcs_nss_supp;
	uint8_t					ppe_thres[IEEE80211_HE_CAP_PPE_THRES_MAX];
};

/* 802.11ax-2021, 9.4.2.249 HE Operation element. */
#define	IEEE80211_HE_OPERATION_ER_SU_DISABLE		(1<<16)
#define	IEEE80211_HE_OPERATION_BSS_COLOR_OFFSET		24
#define	IEEE80211_HE_OPERATION_BSS_COLOR_DISABLED	(1<<31)

struct ieee80211_he_operation {
	uint32_t				he_oper_params;	/* (3) params | (1) bss color info */
	uint16_t				he_mcs_nss_set;
	uint8_t					optional[0];
} __packed;

/* 802.11ax-2021, 9.4.2.251 MU EDCA Parameter Set element. */
struct ieee80211_he_mu_edca_param_ac_rec {
	uint8_t					aifsn;
	uint8_t					ecw_min_max;
	uint8_t					mu_edca_timer;
} __packed;

struct ieee80211_mu_edca_param_set {
	uint8_t					mu_qos_info;
	union {
		struct {
			struct ieee80211_he_mu_edca_param_ac_rec ac_be;
			struct ieee80211_he_mu_edca_param_ac_rec ac_bk;
			struct ieee80211_he_mu_edca_param_ac_rec ac_vi;
			struct ieee80211_he_mu_edca_param_ac_rec ac_vo;
		};
		struct ieee80211_he_mu_edca_param_ac_rec param_ac_recs[4];
	};
} __packed;

/* 802.11ax-2021, 9.4.2.252 Spatial Reuse Parameter Set element */
/* Figure 9-788r-SR Control field format */
#define	IEEE80211_HE_SPR_NON_SRG_OBSS_PD_SR_DISALLOWED	(1<<1)
#define	IEEE80211_HE_SPR_NON_SRG_OFFSET_PRESENT		(1<<2)
#define	IEEE80211_HE_SPR_SRG_INFORMATION_PRESENT	(1<<3)
#define	IEEE80211_HE_SPR_HESIGA_SR_VAL15_ALLOWED	(1<<4)

/* 802.11ax-2021, 9.4.2.263 HE 6 GHz Band Capabilities element */
/* Figure 9-788aj-Capabilities Information field format */
#define	IEEE80211_HE_6GHZ_CAP_MIN_MPDU_START		(1<<2 | 1<<1 | 1<<0)
#define	IEEE80211_HE_6GHZ_CAP_MAX_AMPDU_LEN_EXP		(1<<5 | 1<<4 | 1<<3)
#define	IEEE80211_HE_6GHZ_CAP_MAX_MPDU_LEN		(1<<7 | 1<<6)
#define	IEEE80211_HE_6GHZ_CAP_SM_PS			(1<<10 | 1<<9)
#define	IEEE80211_HE_6GHZ_CAP_RX_ANTPAT_CONS		(1<<12)
#define	IEEE80211_HE_6GHZ_CAP_TX_ANTPAT_CONS		(1<<13)

struct ieee80211_he_6ghz_capa {
	uint16_t capa;
};
#endif	/* _KERNEL || WANT_NET80211 */

/*
 * Management information element payloads.
 *
 * 802.11-2020 Table 9-92 (Element IDs).
 * 802.11ax-2021
 */

enum {
	IEEE80211_ELEMID_SSID		= 0,
	IEEE80211_ELEMID_RATES		= 1,
	/* 2 Reserved */
	IEEE80211_ELEMID_FHPARMS	= 2,	/* remove? */
	IEEE80211_ELEMID_DSPARMS	= 3,
	/* 4 Reserved */
	IEEE80211_ELEMID_CFPARMS	= 4,	/* remove? */
	IEEE80211_ELEMID_TIM		= 5,
	IEEE80211_ELEMID_IBSSPARMS	= 6,
	IEEE80211_ELEMID_COUNTRY	= 7,
	/* 8, 9 reserved */
	IEEE80211_ELEMID_REQUEST	= 10,
	IEEE80211_ELEMID_BSSLOAD	= 11,
	IEEE80211_ELEMID_EDCA_PARAM_SET	= 12,
	IEEE80211_ELEMID_TSPEC		= 13,
	IEEE80211_ELEMID_TCLAS		= 14,
	IEEE80211_ELEMID_SCHEDULE	= 15,
	IEEE80211_ELEMID_CHALLENGE	= 16,
	/* 17-31 reserved for challenge text extension */
	IEEE80211_ELEMID_PWRCNSTR	= 32,
	IEEE80211_ELEMID_PWRCAP		= 33,
	IEEE80211_ELEMID_TPCREQ		= 34,
	IEEE80211_ELEMID_TPCREP		= 35,
	IEEE80211_ELEMID_SUPPCHAN	= 36,
	IEEE80211_ELEMID_CSA		= 37,
	IEEE80211_ELEMID_MEASREQ	= 38,
	IEEE80211_ELEMID_MEASREP	= 39,
	IEEE80211_ELEMID_QUIET		= 40,
	IEEE80211_ELEMID_IBSSDFS	= 41,
	IEEE80211_ELEMID_ERP		= 42,
	IEEE80211_ELEMID_TS_DELAY	= 43,
	IEEE80211_ELEMID_TCLAS_PROCES	= 44,
	IEEE80211_ELEMID_HTCAP		= 45,
	IEEE80211_ELEMID_QOS		= 46,
	/* 47 reserved */
	IEEE80211_ELEMID_RESERVED_47	= 47,
	IEEE80211_ELEMID_RSN		= 48,
	/* 49 reserved */
	IEEE80211_ELEMID_XRATES		= 50,
	IEEE80211_ELEMID_APCHANREP	= 51,
	IEEE80211_ELEMID_NEIGHBOR_REP	= 52,
	IEEE80211_ELEMID_RCPI		= 53,
	IEEE80211_ELEMID_MOBILITY_DOMAIN = 54,	/* MDE */
	IEEE80211_ELEMID_FAST_BSS_TRANS	= 55,	/* FTE */
	IEEE80211_ELEMID_TIMEOUT_INTVL	= 56,
	IEEE80211_ELEMID_RIC_DATA	= 57,	/* RDE */
	IEEE80211_ELEMID_DSE_REG_LOC	= 58,
	IEEE80211_ELEMID_SUP_OP_CLASS	= 59,
	IEEE80211_ELEMID_EXT_CSA	= 60,
	IEEE80211_ELEMID_HTINFO		= 61,	/* HTOPER */
	IEEE80211_ELEMID_SECCHAN_OFFSET	= 62,
	IEEE80211_ELEMID_BSS_AVG_ACC_DELAY = 63,
	IEEE80211_ELEMID_ANTENNA	= 64,
	IEEE80211_ELEMID_RSNI		= 65,
	IEEE80211_ELEMID_MEAS_PILOT_TRANS = 66,
	IEEE80211_ELEMID_BSS_AVAIL_AD_CAP = 67,
	IEEE80211_ELEMID_BSS_AC_ACC_DELAY = 68,
	IEEE80211_ELEMID_TIME_ADV	= 69,
	IEEE80211_ELEMID_RRM_ENACAPS	= 70,	/* RM_ENCAPS */
	IEEE80211_ELEMID_MULTIBSSID	= 71,
	IEEE80211_ELEMID_COEX_2040	= 72,
	IEEE80211_ELEMID_2040_INTOL_CHAN_REPORT	= 73,
	IEEE80211_ELEMID_OVERLAP_BSS_SCAN_PARAM = 74,
	IEEE80211_ELEMID_RIC_DESC	= 75,
	IEEE80211_ELEMID_MGMT_MIC	= 76,
	/* 77 reserved */
	IEEE80211_ELEMID_EVENT_REQ	= 78,
	IEEE80211_ELEMID_EVENT_REP	= 79,
	IEEE80211_ELEMID_DIAGNOSTIC_REQ	= 80,
	IEEE80211_ELEMID_DIAGNOSTIC_REP	= 81,
	IEEE80211_ELEMID_LOCATION_PARAM	= 82,
	IEEE80211_ELEMID_NONTRANS_BSSID_CAP = 83,
	IEEE80211_ELEMID_SSID_LIST	= 84,
	IEEE80211_ELEMID_MULTI_BSSID_IDX = 85,
	IEEE80211_ELEMID_FMS_DESC	= 86,
	IEEE80211_ELEMID_FMS_REQ	= 87,
	IEEE80211_ELEMID_FMS_RESP	= 88,
	IEEE80211_ELEMID_QOS_TRAFFIC_CAP = 89,
	IEEE80211_ELEMID_BSS_MAX_IDLE_P	= 90,
	IEEE80211_ELEMID_TSF_REQ	= 91,
	IEEE80211_ELEMID_TSF_RESP	= 92,
	IEEE80211_ELEMID_WNM_SLEEP_MODE	= 93,
	IEEE80211_ELEMID_TIM_BCAST_REQ	= 94,
	IEEE80211_ELEMID_TIM_BCAST_RESP	= 95,
	IEEE80211_ELEMID_COL_INTERF_REP = 96,
	IEEE80211_ELEMID_CHAN_USAGE	= 97,
	IEEE80211_ELEMID_TIME_ZONE	= 98,
	IEEE80211_ELEMID_DMS_REQ	= 99,
	IEEE80211_ELEMID_DMS_RESP	= 100,
	IEEE80211_ELEMID_LINK_ID	= 101,
	IEEE80211_ELEMID_WAKEUP_SCHED	= 102,
	/* 103 reserved */
	IEEE80211_ELEMID_CHAN_SW_TIMING	= 104,
	IEEE80211_ELEMID_PTI_CONTROL	= 105,
	IEEE80211_ELEMID_TPU_BUF_STATUS	= 106,
	IEEE80211_ELEMID_INTERWORKING	= 107,
	IEEE80211_ELEMID_ADV_PROTO	= 108,
	IEEE80211_ELEMID_EXP_BW_REQ	= 109,
	IEEE80211_ELEMID_QOS_MAP	= 110,
	IEEE80211_ELEMID_ROAM_CONSORT	= 111,
	IEEE80211_ELEMID_EMERG_ALERT_ID	= 112,

	/*
	 * 802.11s IEs
	 * NB: On vanilla Linux still IEEE80211_ELEMID_MESHPEER = 55,
	 * but they defined a new with id 117 called PEER_MGMT.
	 * NB: complies with open80211
	 */
	IEEE80211_ELEMID_MESHCONF	= 113,
	IEEE80211_ELEMID_MESHID		= 114,
	IEEE80211_ELEMID_MESHLINK	= 115,
	IEEE80211_ELEMID_MESHCNGST	= 116,
	IEEE80211_ELEMID_MESHPEER	= 117,
	IEEE80211_ELEMID_MESHCSA	= 118,
	IEEE80211_ELEMID_MESHTIM	= 39, /* XXX: remove */
	IEEE80211_ELEMID_MESHAWAKEW	= 119,
	IEEE80211_ELEMID_MESHBEACONT	= 120,
	/* 121-124 MMCAOP not implemented yet */
	IEEE80211_ELEMID_MESHGANN	= 125,
	IEEE80211_ELEMID_MESHRANN	= 126,
	/* 127 Extended Capabilities */
	IEEE80211_ELEMID_EXTCAP		= 127,
	/* 128-129 reserved */
	IEEE80211_ELEMID_MESHPREQ	= 130,
	IEEE80211_ELEMID_MESHPREP	= 131,
	IEEE80211_ELEMID_MESHPERR	= 132,
	/* 133-136 reserved */
	IEEE80211_ELEMID_MESHPXU	= 137,
	IEEE80211_ELEMID_MESHPXUC	= 138,
	IEEE80211_ELEMID_MESHAH		= 60, /* XXX: remove */
	/* XXX 139 */

	IEEE80211_ELEMID_MIC		= 140,
	IEEE80211_ELEMID_DEST_URI	= 141,
	IEEE80211_ELEMID_U_APSD_COEX	= 142,
	IEEE80211_ELEMID_DMG_WAKEUP_SCHED = 143,
	IEEE80211_ELEMID_EXT_SCHED	= 144,
	IEEE80211_ELEMID_STA_AVAIL	= 145,
	IEEE80211_ELEMID_DMG_TSPEC	= 146,
	IEEE80211_ELEMID_NEXT_DMG_ATI	= 147,
	IEEE80211_ELEMID_DMG_CAP	= 148,
	/* 149-150 reserved. */
	IEEE80211_ELEMID_TPC		= 150, /* XXX: remove */
	IEEE80211_ELEMID_DMG_OPER	= 151,
	IEEE80211_ELEMID_DMG_BSS_PARAM_CHANGE = 152,
	IEEE80211_ELEMID_DMG_BEAM_REF	= 153,
	IEEE80211_ELEMID_CHAN_MEAS_FEEDB = 154,
	/* 155-156 reserved. */
	IEEE80211_ELEMID_CCKM		= 156, /* XXX: remove? */
	IEEE80211_ELEMID_AWAKE_WIN	= 157,
	IEEE80211_ELEMID_MULTI_BAND	= 158,
	IEEE80211_ELEMID_ADDBA_EXT	= 159,
	IEEE80211_ELEMID_NEXTPCP_LIST	= 160,
	IEEE80211_ELEMID_PCP_HANDOVER	= 161,
	IEEE80211_ELEMID_DMG_LINK_MARGIN = 162,
	IEEE80211_ELEMID_SW_STREAM	= 163,
	IEEE80211_ELEMID_SESS_TRANS	= 164,
	/* 165 reserved. */
	IEEE80211_ELEMID_CLUSTER_REP	= 166,
	IEEE80211_ELEMID_RELAY_CAP	= 167,
	IEEE80211_ELEMID_RELAY_TRANS_PARAM_SET = 168,
	IEEE80211_ELEMID_BEAMLINK_MAINT	= 169,
	IEEE80211_ELEMID_MULTI_MAC_SUBL	= 170,
	IEEE80211_ELEMID_U_PID		= 171,
	IEEE80211_ELEMID_DMG_LINK_ADAP_ACK = 172,
	/* 173 reserved. */
	IEEE80211_ELEMID_MCCAOP_ADV_OV	= 174,
	IEEE80211_ELEMID_QUIET_PERIOD_REQ = 175,
	/* 176 reserved. */
	IEEE80211_ELEMID_QUIET_PERIOD_RESP = 177,
	/* 178-180 reserved. */
	IEEE80211_ELEMID_QMF_POLICY	= 181,
	IEEE80211_ELEMID_ECAPC_POLICY	= 182,
	IEEE80211_ELEMID_CLUSTER_TIME_OFFSET = 183,
	IEEE80211_ELEMID_INTRA_ACC_CAT_PRIO = 184,
	IEEE80211_ELEMID_SCS_DESCR	= 185,
	IEEE80211_ELEMID_QLOAD_REPORT	= 186,
	IEEE80211_ELEMID_HCCA_TXOP_UP_CNT = 187,
	IEEE80211_ELEMID_HL_STREAM_ID	= 188,
	IEEE80211_ELEMID_GCR_GROUP_ADDR	= 189,
	IEEE80211_ELEMID_ANTENNA_SECTOR_ID_PAT = 190,
	IEEE80211_ELEMID_VHT_CAP	= 191,
	IEEE80211_ELEMID_VHT_OPMODE	= 192,
	IEEE80211_ELEMID_EXTENDED_BSS_LOAD = 193,
	IEEE80211_ELEMID_WIDE_BW_CHAN_SW = 194,
	IEEE80211_ELEMID_VHT_PWR_ENV	= 195,	/* TX_PWR_ENV */
	IEEE80211_ELEMID_CHANNEL_SWITCH_WRAPPER = 196,
	IEEE80211_ELEMID_AID		= 197,
	IEEE80211_ELEMID_QUIET_CHANNEL	= 198,
	IEEE80211_ELEMID_OPMODE_NOTIF	= 199,
	IEEE80211_ELEMID_UPSIM		= 200,
	IEEE80211_ELEMID_RED_NEIGH_REP	= 201,
	IEEE80211_ELEMID_TVHT_OP	= 202,
	/* 203 reserved. */
	IEEE80211_ELEMID_DEVICE_LOC	= 204,
	IEEE80211_ELEMID_WHITE_SPACE_MAP = 205,
	IEEE80211_ELEMID_FINE_TIME_MEAS_PARAM = 206,
	IEEE80211_ELEMID_SIG_OPEN_LOOP_LINK_MARGIN_IDX	= 207,
	IEEE80211_ELEMID_RPS		= 208,
	IEEE80211_ELEMID_PAGE_SLICE	= 209,
	IEEE80211_ELEMID_AID_REQ	= 210,
	IEEE80211_ELEMID_AID_RESP	= 211,
	IEEE80211_ELEMID_SIG_SECTOR_OP	= 212,
	IEEE80211_ELEMID_SIG_BEACON_COMPAT = 213,
	IEEE80211_ELEMID_SHORT_BEACON_INTVL = 214,
	IEEE80211_ELEMID_CHANGE_SEQ	= 215,
	IEEE80211_ELEMID_TWT		= 216,
	IEEE80211_ELEMID_SIG_CAPS	= 217,
	/* 218-219 reserved. */
	IEEE80211_ELEMID_SUBCHAN_SELECT_TRANS = 220,
	IEEE80211_ELEMID_VENDOR		= 221,	/* vendor private */
	IEEE80211_ELEMID_AUTH_CONTROL	= 222,
	IEEE80211_ELEMID_TSF_TIMER_ACC	= 223,
	IEEE80211_ELEMID_S1G_RELAY	= 224,
	IEEE80211_ELEMID_REACHABLE_ADDR	= 225,
	IEEE80211_ELEMID_SIG_RELAY_DISC	= 226,
	/* 227 reserved. */
	IEEE80211_ELEMID_AID_ANNOUNCEMENT = 228,
	IEEE80211_ELEMID_PV1_PROBE_RESP_OPT = 229,
	IEEE80211_ELEMID_EL_OP		= 230,
	IEEE80211_ELEMID_SECTORIZED_GRP_ID_LIST	= 231,
	IEEE80211_ELEMID_SIG_OP		= 232,
	IEEE80211_ELEMID_HDR_COMPRESSION = 233,
	IEEE80211_ELEMID_SST_OP		= 234,
	IEEE80211_ELEMID_MAD		= 235,
	IEEE80211_ELEMID_SIG_RELAY_ACT	= 236,
	IEEE80211_ELEMID_CAG_NUMBER	= 237,
	/* 238 reserved. */
	IEEE80211_ELEMID_AP_CSN		= 239,
	IEEE80211_ELEMID_FILS_INDICATION = 240,
	IEEE80211_ELEMID_DILS		= 241,
	IEEE80211_ELEMID_FRAGMENT	= 242,
	/* 243 reserved. */
	IEEE80211_ELEMID_RSN_EXT	= 244,
	/* 245-254 reserved. */
	IEEE80211_ELEMID_EXTFIELD	= 255
};

enum ieee80211_elemid_ext {
	IEEE80211_ELEMID_EXT_ASSOC_DELAY_INFO		= 1,
	IEEE80211_ELEMID_EXT_FILS_REQ_PARAMS		= 2,
	IEEE80211_ELEMID_EXT_FILS_KEY_CONFIRM		= 3,
	IEEE80211_ELEMID_EXT_FILS_SESSION		= 4,
	IEEE80211_ELEMID_EXT_FILS_HLP_CONTAINER		= 5,
	IEEE80211_ELEMID_EXT_FILS_IP_ADDR_ASSIGNMENT	= 6,
	IEEE80211_ELEMID_EXT_KEY_DELIVERY		= 7,
	IEEE80211_ELEMID_EXT_FILS_WRAPPED_DATA		= 8,
	IEEE80211_ELEMID_EXT_FTM_SYNC_INFO		= 9,
	IEEE80211_ELEMID_EXT_EXT_REQ			= 10,
	IEEE80211_ELEMID_EXT_EST_SERVICE_PARAM_INBOUND	= 11,
	IEEE80211_ELEMID_EXT_FILS_PUBLIC_KEY		= 12,
	IEEE80211_ELEMID_EXT_FILS_NONCE			= 13,
	IEEE80211_ELEMID_EXT_FUTURE_CHAN_GUIDANCE	= 14,
	IEEE80211_ELEMID_EXT_SERVICE_HINT		= 15,
	IEEE80211_ELEMID_EXT_SERVICE_HASH		= 16,
	IEEE80211_ELEMID_EXT_CDMG_CAPA			= 17,
	IEEE80211_ELEMID_EXT_DYN_BW_CTRL		= 18,
	IEEE80211_ELEMID_EXT_CDMG_EXT_SCHEDULE		= 19,
	IEEE80211_ELEMID_EXT_SSW_REPORT			= 20,
	IEEE80211_ELEMID_EXT_CLUSTER_PROBE		= 21,
	IEEE80211_ELEMID_EXT_EXT_CLUSTER_REPORT		= 22,
	IEEE80211_ELEMID_EXT_CLUSTER_SW_ANNOUNCEMENT	= 23,
	IEEE80211_ELEMID_EXT_ENHANCED_BEAM_TRACKING	= 24,
	IEEE80211_ELEMID_EXT_SPSH_REPORT		= 25,
	IEEE80211_ELEMID_EXT_CLUSTER_INTERF_ASSESS	= 26,
	IEEE80211_ELEMID_EXT_CMMG_CAPA			= 27,
	IEEE80211_ELEMID_EXT_CMMG_OPER			= 28,
	IEEE80211_ELEMID_EXT_CMMG_OPMODE_NOTIF		= 29,
	IEEE80211_ELEMID_EXT_CMMG_LINK_MARGIN		= 30,
	IEEE80211_ELEMID_EXT_CMMG_LINK_ADAP_ACK		= 31,
	/* 32 reserved. */
	IEEE80211_ELEMID_EXT_PASSWORD_ID		= 33,
	IEEE80211_ELEMID_EXT_GLK_GCR_PARAM_SET		= 34,
	IEEE80211_ELEMID_EXT_HE_CAPA			= 35,
	IEEE80211_ELEMID_EXT_HE_OPER			= 36,
	IEEE80211_ELEMID_EXT_UORA_PARAM_SET		= 37,
	IEEE80211_ELEMID_EXT_MU_EDCA_PARAM_SET		= 38,
	IEEE80211_ELEMID_EXT_SPATIAL_REUSE_PARAM_SET	= 39,
	IEEE80211_ELEMID_EXT_GAS_EXTENSION		= 40,
	IEEE80211_ELEMID_EXT_NDP_FEEDB_REPORT_PARAM	= 41,
	IEEE80211_ELEMID_EXT_BSS_COLOR_CHG_ANNOUNCE	= 42,
	IEEE80211_ELEMID_EXT_QUIET_TIMME_PERIOD		= 43,
	IEEE80211_ELEMID_EXT_VENDOR_SPECIFIC_REQ_ELEM	= 44,
	IEEE80211_ELEMID_EXT_ESS_REPORT			= 45,
	IEEE80211_ELEMID_EXT_OPS			= 46,
	IEEE80211_ELEMID_EXT_HE_BSS_LOAD		= 47,
	/* 48-51 reserved. */
	IEEE80211_ELEMID_EXT_MAC_CH_SW_TIME		= 52,
	IEEE80211_ELEMID_EXT_EST_SERVICE_PARAM_OUTBOUND	= 53,
	IEEE80211_ELEMID_EXT_OCI			= 54,
	IEEE80211_ELEMID_EXT_MULTI_BSSID_CONFIG		= 55,
	IEEE80211_ELEMID_EXT_NON_INHERITANCE		= 56,
	IEEE80211_ELEMID_EXT_KNOWN_BSSID		= 57,
	IEEE80211_ELEMID_EXT_SHORT_SSID_LIST		= 58,
	IEEE80211_ELEMID_EXT_HE_6GHZ_BAND_CAPA		= 59,
	IEEE80211_ELEMID_EXT_ULMU_POWER_CAAP		= 60,
	/* 61-87 reserved. */
	IEEE80211_ELEMID_EXT_MSCS_DESCRIPTOR		= 88,
	IEEE80211_ELEMID_EXT_TCLAS_MASK			= 89,
	IEEE80211_ELEMID_EXT_SUPPL_CLASS_2_CAPA		= 90,
	IEEE80211_ELEMID_EXT_OCT_SOURCE			= 91,
	IEEE80211_ELEMID_EXT_REJECTED_GROUPS		= 92,
	IEEE80211_ELEMID_EXT_ANTI_CLOGGING_TAOKEN_CONTAINER = 93,
	/* 94-255 reserved. */
};

struct ieee80211_tim_ie {
	uint8_t		tim_ie;			/* IEEE80211_ELEMID_TIM */
	uint8_t		tim_len;
	uint8_t		tim_count;		/* DTIM count */
	uint8_t		tim_period;		/* DTIM period */
	uint8_t		tim_bitctl;		/* bitmap control */
	uint8_t		tim_bitmap[1];		/* variable-length bitmap */
} __packed;

struct ieee80211_country_ie {
	uint8_t		ie;			/* IEEE80211_ELEMID_COUNTRY */
	uint8_t		len;
	uint8_t		cc[3];			/* ISO CC+(I)ndoor/(O)utdoor */
	struct {
		uint8_t schan;			/* starting channel */
		uint8_t nchan;			/* number channels */
		uint8_t maxtxpwr;		/* tx power cap */
	} __packed band[1];			/* sub bands (NB: var size) */
} __packed;

#define	IEEE80211_COUNTRY_MAX_BANDS	84	/* max possible bands */
#define	IEEE80211_COUNTRY_MAX_SIZE \
	(sizeof(struct ieee80211_country_ie) + 3*(IEEE80211_COUNTRY_MAX_BANDS-1))

struct ieee80211_bss_load_ie {
	uint8_t		ie;
	uint8_t		len;
	uint16_t	sta_count;	/* station count */
	uint8_t		chan_load;	/* channel utilization */
	uint8_t		aac;		/* available admission capacity */
} __packed;

struct ieee80211_ap_chan_report_ie {
	uint8_t		ie;
	uint8_t		len;
	uint8_t		i_class; /* operating class */
	/* Annex E, E.1 Country information and operating classes */
	uint8_t		chan_list[0];
} __packed;

#define IEEE80211_EXTCAP_CMS			(1ULL <<  0) /* 20/40 BSS coexistence management support */
#define IEEE80211_EXTCAP_RSVD_1			(1ULL <<  1)
#define IEEE80211_EXTCAP_ECS			(1ULL <<  2) /* extended channel switching */
#define IEEE80211_EXTCAP_RSVD_3			(1ULL <<  3)
#define IEEE80211_EXTCAP_PSMP_CAP		(1ULL <<  4) /* PSMP capability */
#define IEEE80211_EXTCAP_RSVD_5			(1ULL <<  5)
#define IEEE80211_EXTCAP_S_PSMP_SUPP		(1ULL <<  6)
#define IEEE80211_EXTCAP_EVENT			(1ULL <<  7)
#define IEEE80211_EXTCAP_DIAGNOSTICS		(1ULL <<  8)
#define IEEE80211_EXTCAP_MCAST_DIAG		(1ULL <<  9)
#define IEEE80211_EXTCAP_LOC_TRACKING		(1ULL << 10)
#define IEEE80211_EXTCAP_FMS			(1ULL << 11)
#define IEEE80211_EXTCAP_PROXY_ARP		(1ULL << 12)
#define IEEE80211_EXTCAP_CIR			(1ULL << 13) /* collocated interference reporting */
#define IEEE80211_EXTCAP_CIVIC_LOC		(1ULL << 14)
#define IEEE80211_EXTCAP_GEOSPATIAL_LOC		(1ULL << 15)
#define IEEE80211_EXTCAP_TFS			(1ULL << 16)
#define IEEE80211_EXTCAP_WNM_SLEEPMODE		(1ULL << 17)
#define IEEE80211_EXTCAP_TIM_BROADCAST		(1ULL << 18)
#define IEEE80211_EXTCAP_BSS_TRANSITION		(1ULL << 19)
#define IEEE80211_EXTCAP_QOS_TRAF_CAP		(1ULL << 20)
#define IEEE80211_EXTCAP_AC_STA_COUNT		(1ULL << 21)
#define IEEE80211_EXTCAP_M_BSSID		(1ULL << 22) /* multiple BSSID field */
#define IEEE80211_EXTCAP_TIMING_MEAS		(1ULL << 23)
#define IEEE80211_EXTCAP_CHAN_USAGE		(1ULL << 24)
#define IEEE80211_EXTCAP_SSID_LIST		(1ULL << 25)
#define IEEE80211_EXTCAP_DMS			(1ULL << 26)
#define IEEE80211_EXTCAP_UTC_TSF_OFFSET		(1ULL << 27)
#define IEEE80211_EXTCAP_TLDS_BUF_STA_SUPP	(1ULL << 28) /* TDLS peer U-APSP buffer STA support */
#define IEEE80211_EXTCAP_TLDS_PPSM_SUPP		(1ULL << 29) /* TDLS peer PSM support */
#define IEEE80211_EXTCAP_TLDS_CH_SW		(1ULL << 30) /* TDLS channel switching */
#define IEEE80211_EXTCAP_INTERWORKING		(1ULL << 31)
#define IEEE80211_EXTCAP_QOSMAP			(1ULL << 32)
#define IEEE80211_EXTCAP_EBR			(1ULL << 33)
#define IEEE80211_EXTCAP_SSPN_IF		(1ULL << 34)
#define IEEE80211_EXTCAP_RSVD_35		(1ULL << 35)
#define IEEE80211_EXTCAP_MSGCF_CAP		(1ULL << 36)
#define IEEE80211_EXTCAP_TLDS_SUPP		(1ULL << 37)
#define IEEE80211_EXTCAP_TLDS_PROHIB		(1ULL << 38)
#define IEEE80211_EXTCAP_TLDS_CH_SW_PROHIB	(1ULL << 39) /* TDLS channel switching prohibited */
#define IEEE80211_EXTCAP_RUF			(1ULL << 40) /* reject unadmitted frame */
/* service interval granularity */
#define IEEE80211_EXTCAP_SIG \
				((1ULL << 41) | (1ULL << 42) | (1ULL << 43))
#define IEEE80211_EXTCAP_ID_LOC			(1ULL << 44)
#define IEEE80211_EXTCAP_U_APSD_COEX		(1ULL << 45)
#define IEEE80211_EXTCAP_WNM_NOTIFICATION	(1ULL << 46)
#define IEEE80211_EXTCAP_RSVD_47		(1ULL << 47)
#define IEEE80211_EXTCAP_SSID			(1ULL << 48) /* UTF-8 SSID */
/* bits 49-n are reserved */

struct ieee80211_extcap_ie {
	uint8_t		ie;
	uint8_t		len;
} __packed;

/*
 * 802.11h Quiet Time Element.
 */
struct ieee80211_quiet_ie {
	uint8_t		quiet_ie;		/* IEEE80211_ELEMID_QUIET */
	uint8_t		len;
	uint8_t		tbttcount;		/* quiet start */
	uint8_t		period;			/* beacon intervals between quiets */
	uint16_t	duration;		/* TUs of each quiet*/
	uint16_t	offset;			/* TUs of from TBTT of quiet start */
} __packed;

/*
 * 802.11h Channel Switch Announcement (CSA).
 */
struct ieee80211_csa_ie {
	uint8_t		csa_ie;		/* IEEE80211_ELEMID_CHANSWITCHANN */
	uint8_t		csa_len;
	uint8_t		csa_mode;		/* Channel Switch Mode */
	uint8_t		csa_newchan;		/* New Channel Number */
	uint8_t		csa_count;		/* Channel Switch Count */
} __packed;

/*
 * Note the min acceptable CSA count is used to guard against
 * malicious CSA injection in station mode.  Defining this value
 * as other than 0 violates the 11h spec.
 */
#define	IEEE80211_CSA_COUNT_MIN	2
#define	IEEE80211_CSA_COUNT_MAX	255

/* rate set entries are in .5 Mb/s units, and potentially marked as basic */
#define	IEEE80211_RATE_BASIC		0x80
#define	IEEE80211_RATE_VAL		0x7f
#define	IEEE80211_RV(v)			((v) & IEEE80211_RATE_VAL)

/* ERP information element flags */
#define	IEEE80211_ERP_NON_ERP_PRESENT	0x01
#define	IEEE80211_ERP_USE_PROTECTION	0x02
#define	IEEE80211_ERP_LONG_PREAMBLE	0x04

#define	IEEE80211_ERP_BITS \
	"\20\1NON_ERP_PRESENT\2USE_PROTECTION\3LONG_PREAMBLE"

#define	ATH_OUI			0x7f0300	/* Atheros OUI */
#define	ATH_OUI_TYPE		0x01		/* Atheros protocol ie */

/* NB: Atheros allocated the OUI for this purpose ~2005 but beware ... */
#define	TDMA_OUI		ATH_OUI
#define	TDMA_OUI_TYPE		0x02		/* TDMA protocol ie */

#define	BCM_OUI			0x4c9000	/* Broadcom OUI */
#define	BCM_OUI_HTCAP		51		/* pre-draft HTCAP ie */
#define	BCM_OUI_HTINFO		52		/* pre-draft HTINFO ie */

#define	WPA_OUI			0xf25000
#define	WPA_OUI_TYPE		0x01
#define	WPA_VERSION		1		/* current supported version */

#define	WPA_CSE_NULL		0x00
#define	WPA_CSE_WEP40		0x01
#define	WPA_CSE_TKIP		0x02
#define	WPA_CSE_CCMP		0x04		/* CCMP 128-bit */
#define	WPA_CSE_WEP104		0x05

#define	WPA_ASE_NONE		0x00
#define	WPA_ASE_8021X_UNSPEC	0x01
#define	WPA_ASE_8021X_PSK	0x02

#define	WPS_OUI_TYPE		0x04

/* 802.11-2016 Table 9-131 - Cipher Suite Selectors */
#define	RSN_OUI			0xac0f00
#define	RSN_VERSION		1		/* current supported version */

/* RSN cipher suite element */
#define	RSN_CSE_NULL		0
#define	RSN_CSE_WEP40		1
#define	RSN_CSE_TKIP		2
#define	RSN_CSE_WRAP		3		/* Reserved in the 802.11-2016 */
#define	RSN_CSE_CCMP		4		/* CCMP 128 bit */
#define	RSN_CSE_WEP104		5
#define	RSN_CSE_BIP_CMAC_128	6
/* 7 - "Group addressed traffic not allowed" */
#define	RSN_CSE_GCMP_128	8
#define	RSN_CSE_GCMP_256	9
#define	RSN_CSE_CCMP_256	10
#define	RSN_CSE_BIP_GMAC_128	11
#define	RSN_CSE_BIP_GMAC_256	12
#define	RSN_CSE_BIP_CMAC_256	13

/* 802.11-2016 Table 9-133 - AKM suite selectors */
/* RSN AKM suite element */
#define	RSN_ASE_NONE		0
#define	RSN_ASE_8021X_UNSPEC	1
#define	RSN_ASE_8021X_PSK	2
#define	RSN_ASE_FT_8021X	3		/* SHA-256 */
#define	RSN_ASE_FT_PSK		4		/* SHA-256 */
#define	RSN_ASE_8021X_UNSPEC_SHA256	5
#define	RSN_ASE_8021X_PSK_SHA256	6
#define	RSN_ASE_8021X_TDLS	7		/* SHA-256 */
#define	RSN_ASE_SAE_UNSPEC	8		/* SHA-256 */
#define	RSN_ASE_FT_SAE		9		/* SHA-256 */
#define	RSN_ASE_AP_PEERKEY	10		/* SHA-256 */
#define	RSN_ASE_8021X_SUITE_B_SHA256	11
#define	RSN_ASE_8021X_SUITE_B_SHA384	12
#define	RSN_ASE_FT_8021X_SHA384	13

/* 802.11-2016 Figure 9-257 - RSN Capabilities (2 byte field) */
#define	RSN_CAP_PREAUTH		0x0001
#define	RSN_CAP_NO_PAIRWISE	0x0002
#define	RSN_CAP_PTKSA_REPLAY_COUNTER	0x000c	/* 2 bit field */
#define	RSN_CAP_GTKSA_REPLAY_COUNTER	0x0030	/* 2 bit field */
#define	RSN_CAP_MFP_REQUIRED	0x0040
#define	RSN_CAP_MFP_CAPABLE	0x0080
#define	RSN_CAP_JOINT_MULTIBAND_RSNA		0x0100
#define	RSN_CAP_PEERKEY_ENABLED	0x0200
#define	RSN_CAP_SPP_AMSDU_CAPABLE	0x0400
#define	RSN_CAP_SPP_AMSDU_REQUIRED	0x0800
#define	RSN_CAP_PBAC_CAPABLE	0x1000
#define	RSN_CAP_EXT_KEYID_CAPABLE	0x0200

/* 802.11-2016 Table 9-134 PTKSA/GTKSA/STKSA replay counters usage */
#define		RSN_CAP_REPLAY_COUNTER_1_PER	0
#define		RSN_CAP_REPLAY_COUNTER_2_PER	1
#define		RSN_CAP_REPLAY_COUNTER_4_PER	2
#define		RSN_CAP_REPLAY_COUNTER_16_PER	3

#define	WME_OUI			0xf25000
#define	WME_OUI_TYPE		0x02
#define	WME_INFO_OUI_SUBTYPE	0x00
#define	WME_PARAM_OUI_SUBTYPE	0x01
#define	WME_VERSION		1

/* WME stream classes */
#define	WME_AC_BE	0		/* best effort */
#define	WME_AC_BK	1		/* background */
#define	WME_AC_VI	2		/* video */
#define	WME_AC_VO	3		/* voice */

/*
 * AUTH management packets
 *
 *	octet algo[2]
 *	octet seq[2]
 *	octet status[2]
 *	octet chal.id
 *	octet chal.length
 *	octet chal.text[253]		NB: 1-253 bytes
 */

/* challenge length for shared key auth */
#define IEEE80211_CHALLENGE_LEN		128

#define	IEEE80211_AUTH_ALG_OPEN		0x0000
#define	IEEE80211_AUTH_ALG_SHARED	0x0001
#define	IEEE80211_AUTH_ALG_LEAP		0x0080

enum {
	IEEE80211_AUTH_OPEN_REQUEST		= 1,
	IEEE80211_AUTH_OPEN_RESPONSE		= 2,
};

enum {
	IEEE80211_AUTH_SHARED_REQUEST		= 1,
	IEEE80211_AUTH_SHARED_CHALLENGE		= 2,
	IEEE80211_AUTH_SHARED_RESPONSE		= 3,
	IEEE80211_AUTH_SHARED_PASS		= 4,
};

/*
 * Reason and status codes.
 *
 * Reason codes are used in management frames to indicate why an
 * action took place (e.g. on disassociation).  Status codes are
 * used in management frames to indicate the result of an operation.
 *
 * Unlisted codes are reserved
 */

enum {
	IEEE80211_REASON_UNSPECIFIED		= 1,
	IEEE80211_REASON_AUTH_EXPIRE		= 2,
	IEEE80211_REASON_AUTH_LEAVE		= 3,
	IEEE80211_REASON_ASSOC_EXPIRE		= 4,
	IEEE80211_REASON_ASSOC_TOOMANY		= 5,
	IEEE80211_REASON_NOT_AUTHED		= 6,
	IEEE80211_REASON_NOT_ASSOCED		= 7,
	IEEE80211_REASON_ASSOC_LEAVE		= 8,
	IEEE80211_REASON_ASSOC_NOT_AUTHED	= 9,
	IEEE80211_REASON_DISASSOC_PWRCAP_BAD	= 10,	/* 11h */
	IEEE80211_REASON_DISASSOC_SUPCHAN_BAD	= 11,	/* 11h */
	IEEE80211_REASON_IE_INVALID		= 13,	/* 11i */
	IEEE80211_REASON_MIC_FAILURE		= 14,	/* 11i */
	IEEE80211_REASON_4WAY_HANDSHAKE_TIMEOUT	= 15,	/* 11i */
	IEEE80211_REASON_GROUP_KEY_UPDATE_TIMEOUT = 16,	/* 11i */
	IEEE80211_REASON_IE_IN_4WAY_DIFFERS	= 17,	/* 11i */
	IEEE80211_REASON_GROUP_CIPHER_INVALID	= 18,	/* 11i */
	IEEE80211_REASON_PAIRWISE_CIPHER_INVALID= 19,	/* 11i */
	IEEE80211_REASON_AKMP_INVALID		= 20,	/* 11i */
	IEEE80211_REASON_UNSUPP_RSN_IE_VERSION	= 21,	/* 11i */
	IEEE80211_REASON_INVALID_RSN_IE_CAP	= 22,	/* 11i */
	IEEE80211_REASON_802_1X_AUTH_FAILED	= 23,	/* 11i */
	IEEE80211_REASON_CIPHER_SUITE_REJECTED	= 24,	/* 11i */
	IEEE80211_REASON_UNSPECIFIED_QOS	= 32,	/* 11e */
	IEEE80211_REASON_INSUFFICIENT_BW	= 33,	/* 11e */
	IEEE80211_REASON_TOOMANY_FRAMES		= 34,	/* 11e */
	IEEE80211_REASON_OUTSIDE_TXOP		= 35,	/* 11e */
	IEEE80211_REASON_LEAVING_QBSS		= 36,	/* 11e */
	IEEE80211_REASON_BAD_MECHANISM		= 37,	/* 11e */
	IEEE80211_REASON_SETUP_NEEDED		= 38,	/* 11e */
	IEEE80211_REASON_TIMEOUT		= 39,	/* 11e */

	IEEE80211_REASON_PEER_LINK_CANCELED	= 52,	/* 11s */
	IEEE80211_REASON_MESH_MAX_PEERS		= 53,	/* 11s */
	IEEE80211_REASON_MESH_CPVIOLATION	= 54,	/* 11s */
	IEEE80211_REASON_MESH_CLOSE_RCVD	= 55,	/* 11s */
	IEEE80211_REASON_MESH_MAX_RETRIES	= 56,	/* 11s */
	IEEE80211_REASON_MESH_CONFIRM_TIMEOUT	= 57,	/* 11s */
	IEEE80211_REASON_MESH_INVALID_GTK	= 58,	/* 11s */
	IEEE80211_REASON_MESH_INCONS_PARAMS	= 59,	/* 11s */
	IEEE80211_REASON_MESH_INVALID_SECURITY	= 60,	/* 11s */
	IEEE80211_REASON_MESH_PERR_NO_PROXY	= 61,	/* 11s */
	IEEE80211_REASON_MESH_PERR_NO_FI	= 62,	/* 11s */
	IEEE80211_REASON_MESH_PERR_DEST_UNREACH	= 63,	/* 11s */
	IEEE80211_REASON_MESH_MAC_ALRDY_EXISTS_MBSS = 64, /* 11s */
	IEEE80211_REASON_MESH_CHAN_SWITCH_REG	= 65,	/* 11s */
	IEEE80211_REASON_MESH_CHAN_SWITCH_UNSPEC = 66,	/* 11s */

	IEEE80211_STATUS_SUCCESS		= 0,
	IEEE80211_STATUS_UNSPECIFIED		= 1,
	IEEE80211_STATUS_CAPINFO		= 10,
	IEEE80211_STATUS_NOT_ASSOCED		= 11,
	IEEE80211_STATUS_OTHER			= 12,
	IEEE80211_STATUS_ALG			= 13,
	IEEE80211_STATUS_SEQUENCE		= 14,
	IEEE80211_STATUS_CHALLENGE		= 15,
	IEEE80211_STATUS_TIMEOUT		= 16,
	IEEE80211_STATUS_TOOMANY		= 17,
	IEEE80211_STATUS_BASIC_RATE		= 18,
	IEEE80211_STATUS_SP_REQUIRED		= 19,	/* 11b */
	IEEE80211_STATUS_PBCC_REQUIRED		= 20,	/* 11b */
	IEEE80211_STATUS_CA_REQUIRED		= 21,	/* 11b */
	IEEE80211_STATUS_SPECMGMT_REQUIRED	= 22,	/* 11h */
	IEEE80211_STATUS_PWRCAP_REQUIRED	= 23,	/* 11h */
	IEEE80211_STATUS_SUPCHAN_REQUIRED	= 24,	/* 11h */
	IEEE80211_STATUS_SHORTSLOT_REQUIRED	= 25,	/* 11g */
	IEEE80211_STATUS_DSSSOFDM_REQUIRED	= 26,	/* 11g */
	IEEE80211_STATUS_MISSING_HT_CAPS	= 27,	/* 11n D3.0 */
	IEEE80211_STATUS_INVALID_IE		= 40,	/* 11i */
	IEEE80211_STATUS_GROUP_CIPHER_INVALID	= 41,	/* 11i */
	IEEE80211_STATUS_PAIRWISE_CIPHER_INVALID = 42,	/* 11i */
	IEEE80211_STATUS_AKMP_INVALID		= 43,	/* 11i */
	IEEE80211_STATUS_UNSUPP_RSN_IE_VERSION	= 44,	/* 11i */
	IEEE80211_STATUS_INVALID_RSN_IE_CAP	= 45,	/* 11i */
	IEEE80211_STATUS_CIPHER_SUITE_REJECTED	= 46,	/* 11i */
};

#define	IEEE80211_WEP_KEYLEN		5	/* 40bit */
#define	IEEE80211_WEP_IVLEN		3	/* 24bit */
#define	IEEE80211_WEP_KIDLEN		1	/* 1 octet */
#define	IEEE80211_WEP_CRCLEN		4	/* CRC-32 */
#define	IEEE80211_WEP_TOTLEN		(IEEE80211_WEP_IVLEN + \
					 IEEE80211_WEP_KIDLEN + \
					 IEEE80211_WEP_CRCLEN)
#define	IEEE80211_WEP_NKID		4	/* number of key ids */

/*
 * 802.11i defines an extended IV for use with non-WEP ciphers.
 * When the EXTIV bit is set in the key id byte an additional
 * 4 bytes immediately follow the IV for TKIP.  For CCMP the
 * EXTIV bit is likewise set but the 8 bytes represent the
 * CCMP header rather than IV+extended-IV.
 */
#define	IEEE80211_WEP_EXTIV		0x20
#define	IEEE80211_WEP_EXTIVLEN		4	/* extended IV length */
#define	IEEE80211_WEP_MICLEN		8	/* trailing MIC */

#define	IEEE80211_CRC_LEN		4

/*
 * Maximum acceptable MTU is:
 *	IEEE80211_MAX_LEN - WEP overhead - CRC -
 *		QoS overhead - RSN/WPA overhead
 * Min is arbitrarily chosen > IEEE80211_MIN_LEN.  The default
 * mtu is Ethernet-compatible; it's set by ether_ifattach.
 */
#define	IEEE80211_MTU_MAX		2290
#define	IEEE80211_MTU_MIN		32

#define	IEEE80211_MAX_LEN		(2300 + IEEE80211_CRC_LEN + \
    (IEEE80211_WEP_IVLEN + IEEE80211_WEP_KIDLEN + IEEE80211_WEP_CRCLEN))
#define	IEEE80211_ACK_LEN \
	(sizeof(struct ieee80211_frame_ack) + IEEE80211_CRC_LEN)
#define	IEEE80211_MIN_LEN \
	(sizeof(struct ieee80211_frame_min) + IEEE80211_CRC_LEN)

/*
 * The 802.11 spec says at most 2007 stations may be
 * associated at once.  For most AP's this is way more
 * than is feasible so we use a default of IEEE80211_AID_DEF.
 * This number may be overridden by the driver and/or by
 * user configuration but may not be less than IEEE80211_AID_MIN
 * (see _ieee80211.h for implementation-specific settings).
 */
#define	IEEE80211_AID_MAX		2007

#define	IEEE80211_AID(b)	((b) &~ 0xc000)

/* 
 * RTS frame length parameters.  The default is specified in
 * the 802.11 spec as 512; we treat it as implementation-dependent
 * so it's defined in ieee80211_var.h.  The max may be wrong
 * for jumbo frames.
 */
#define	IEEE80211_RTS_MIN		1
#define	IEEE80211_RTS_MAX		2346

/* 
 * TX fragmentation parameters.  As above for RTS, we treat
 * default as implementation-dependent so define it elsewhere.
 */
#define	IEEE80211_FRAG_MIN		256
#define	IEEE80211_FRAG_MAX		2346

/*
 * Beacon interval (TU's).  Min+max come from WiFi requirements.
 * As above, we treat default as implementation-dependent so
 * define it elsewhere.
 */
#define	IEEE80211_BINTVAL_MAX	1000	/* max beacon interval (TU's) */
#define	IEEE80211_BINTVAL_MIN	25	/* min beacon interval (TU's) */

/*
 * DTIM period (beacons).  Min+max are not really defined
 * by the protocol but we want them publicly visible so
 * define them here.
 */
#define	IEEE80211_DTIM_MAX	15	/* max DTIM period */
#define	IEEE80211_DTIM_MIN	1	/* min DTIM period */

/*
 * Beacon miss threshold (beacons).  As for DTIM, we define
 * them here to be publicly visible.  Note the max may be
 * clamped depending on device capabilities.
 */
#define	IEEE80211_HWBMISS_MIN 	1
#define	IEEE80211_HWBMISS_MAX 	255

/*
 * 802.11 frame duration definitions.
 */

struct ieee80211_duration {
	uint16_t	d_rts_dur;
	uint16_t	d_data_dur;
	uint16_t	d_plcp_len;
	uint8_t		d_residue;	/* unused octets in time slot */
};

/* One Time Unit (TU) is 1Kus = 1024 microseconds. */
#define IEEE80211_DUR_TU		1024

/* IEEE 802.11b durations for DSSS PHY in microseconds */
#define IEEE80211_DUR_DS_LONG_PREAMBLE	144
#define IEEE80211_DUR_DS_SHORT_PREAMBLE	72

#define IEEE80211_DUR_DS_SLOW_PLCPHDR	48
#define IEEE80211_DUR_DS_FAST_PLCPHDR	24
#define IEEE80211_DUR_DS_SLOW_ACK	112
#define IEEE80211_DUR_DS_FAST_ACK	56
#define IEEE80211_DUR_DS_SLOW_CTS	112
#define IEEE80211_DUR_DS_FAST_CTS	56

#define IEEE80211_DUR_DS_SLOT		20
#define IEEE80211_DUR_DS_SIFS		10
#define IEEE80211_DUR_DS_PIFS	(IEEE80211_DUR_DS_SIFS + IEEE80211_DUR_DS_SLOT)
#define IEEE80211_DUR_DS_DIFS	(IEEE80211_DUR_DS_SIFS + \
				 2 * IEEE80211_DUR_DS_SLOT)
#define IEEE80211_DUR_DS_EIFS	(IEEE80211_DUR_DS_SIFS + \
				 IEEE80211_DUR_DS_SLOW_ACK + \
				 IEEE80211_DUR_DS_LONG_PREAMBLE + \
				 IEEE80211_DUR_DS_SLOW_PLCPHDR + \
				 IEEE80211_DUR_DIFS)

#endif /* _NET80211_IEEE80211_H_ */